package cz.fidentis.analyst.mesh.core;

import java.util.List;
import javax.vecmath.Vector3d;

/**
 * @author Natália Bebjaková
 * 
 * Represent min-max box.
 * It is automatically maintained by given point array of the model.
 * 
 */

public class BoundingBox {
    private MeshPoint maxPoint;
    private MeshPoint minPoint;
    private MeshPoint midPoint;
    private double maxDiag;
    
    /** 
     * Creates bounding box that is automatically maintained with respect to given array.
     * @param points array of points, must not be null or pempty
     * @throws IllegalArgumentException if the @code{points} param is null or empty
     */
    public BoundingBox(List<MeshPoint> points) {
        if (points == null || points.isEmpty()) {
            throw new IllegalArgumentException("points");
        }
        this.computeMinMax(points);
        this.computeMidDiag();
    }        

    /**
     * 
     * @return max point of the bounding box
     */
    public MeshPoint getMaxPoint() {
        return maxPoint;
    }
    
    /**
     * 
     * @return middle point of the bounding box
     */
    public MeshPoint getMidPoint() {
        return midPoint;
    }
    
    /**
     * 
     * @return min point of the bounding box
     */
    public MeshPoint getMinPoint() {
        return minPoint;
    }
    
    /**
     * Return volume diagonal of the bounding box.
     * @return maximal diagonal of bounding box
     */
    public double getMaxDiag() {
        return maxDiag;
    }
    
    /**
     * Recomputes the BoundingBox from all points
     */
    private void computeMinMax(List<MeshPoint> points) {
        minPoint = new MeshPointImpl(new Vector3d(Double.MAX_VALUE,Double.MAX_VALUE,Double.MAX_VALUE), null, null);
        maxPoint = new MeshPointImpl(new Vector3d(-100000.0,-100000.0,-100000.0), null, null);
 
        for (int i = 0; i < points.size(); i++) {
            MeshPoint point = points.get(i);
            
            minPoint.getPosition().x = Math.min(minPoint.getPosition().x, point.getPosition().x);
            minPoint.getPosition().y = Math.min(minPoint.getPosition().y, point.getPosition().y);
            minPoint.getPosition().z = Math.min(minPoint.getPosition().z, point.getPosition().z);
            
            maxPoint.getPosition().x = Math.max(maxPoint.getPosition().x, point.getPosition().x);
            maxPoint.getPosition().y = Math.max(maxPoint.getPosition().y, point.getPosition().y);
            maxPoint.getPosition().z = Math.max(maxPoint.getPosition().z, point.getPosition().z);                    
        }
    }

    /**
     * Recompute mid-point and max diagonal length.
     */   
    private void computeMidDiag() {
        midPoint = (minPoint.addPosition(maxPoint)).multiplyPosition(0.5);
        MeshPoint diag = maxPoint.subtractPosition(minPoint);
        this.maxDiag = diag.abs();
    }

    /**
     * Returns description of BoundignBox.
     * 
     * @return String representation of the bounding box
     */
    @Override
    public String toString() {
        String str = "BoundingBox: ";
        str += System.lineSeparator();
        str += "\t" + "- min point : " + this.minPoint + System.lineSeparator();
        str += "\t" + "- max point : " + this.maxPoint + System.lineSeparator();
        str += "\t" + "- mid point : " + this.midPoint + System.lineSeparator();
        str += "\t" + "- max diag  : " + this.maxDiag + System.lineSeparator();
        return str;
    }      
}  