package cz.fidentis.analyst.octree;

import cz.fidentis.analyst.mesh.core.MeshFacet;
import cz.fidentis.analyst.mesh.core.MeshPoint;
import java.io.Serializable;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.vecmath.Point3d;

/**
 * A single node of a Octree. Sharing vertices across meshes is supported
 * (the node links all faces that share the same vertex).
 *
 * @author Enkh-Undral EnkhBayar
 */
public class OctNode implements Serializable {
    
    /**
     * 3D location of the node, is null in internal nodes and empty node
     */
    private MeshPoint location;

    /**
     * Mesh facets sharing the stored vertex
     */
    private Map<MeshFacet, Integer> facets = new HashMap<>();

    /**
     * Octree topology
     */
    private List<OctNode> octants;
    
    /**
     * 3D location of the boundary box - corner with smallest coordinates
     */
    private Point3d smallestBoundary;
    
    /**
     * 3D location of the boundary box - corner with largest coordinates
     */
    private Point3d largestBoundary;
    
    /**
     * Constructor of OctNode
     *
     * @param facets Mesh facet containing the mesh vertex. Must not be null nor empty
     * @param indices The index under which the vertex is stored in the mesh facet.
     *              Must be &gt;= 0
     * @param smallest boundary box - corner with smallest coordinates.
     *                 Must not be null
     * @param largest boundary box - corner with largest coordinates.
     *                Must not be null
     * @throws IllegalArgumentException if some parameter is wrong
     */
    public OctNode(List<MeshFacet> facets, List<Integer> indices, Point3d smallest, Point3d largest) {
        if (facets == null || facets.isEmpty()) {
            throw new IllegalArgumentException("facets");
        }
        if (indices == null || indices.isEmpty()) {
            throw new IllegalArgumentException("indices");
        }
        if (facets.size() != indices.size()) {
            throw new IllegalArgumentException("The number of facets and indiecs mismatch");
        }
        if (smallest == null) {
            throw new IllegalArgumentException("Smallest boundary in OctNode cannot be null");
        }
        if (largest == null) {
            throw new IllegalArgumentException("Largest boundary in OctNode cannot be null");
        }
        
        for (int i = 0; i < facets.size(); i++) {
            this.facets.putIfAbsent(facets.get(i), indices.get(i));
        }
        this.location = facets.get(0).getVertex(indices.get(0));
        this.smallestBoundary = smallest;
        this.largestBoundary = largest;
    }
    
    /**
     * Constructor of OctNode for internal nodes
     *
     * @param octants List of octNodes which are children for this node. 
     *                Must not be null and has to have size of 8.
     * @param smallest boundary box - corner with smallest coordinates.
     *                 Must not be null
     * @param largest boundary box - corner with largest coordinates.
     *                Must not be null
     * @throws IllegalArgumentException if some parameter is wrong
     */
    protected OctNode(List<OctNode> octants, Point3d smallest, Point3d largest) {
        if (octants == null || octants.size() != 8) {
            throw new IllegalArgumentException("octants");
        }
        if (smallest == null) {
            throw new IllegalArgumentException("Smallest boundary in OctNode cannot be null");
        }
        if (largest == null) {
            throw new IllegalArgumentException("Largest boundary in OctNode cannot be null");
        }
        
        this.octants = octants;
        this.smallestBoundary = smallest;
        this.largestBoundary = largest;
    }
    
    /**
     * Constructor of OctNode for empty nodes.
     *
     * @param smallest boundary box - corner with smallest coordinates.
     *                 Must not be null
     * @param largest boundary box - corner with largest coordinates.
     *                Must not be null
     * @throws IllegalArgumentException if some parameter is wrong
     */
    protected OctNode(Point3d smallest, Point3d largest) {
        if (smallest == null) {
            throw new IllegalArgumentException("Smallest boundary in OctNode cannot be null");
        }
        if (largest == null) {
            throw new IllegalArgumentException("Largest boundary in OctNode cannot be null");
        }
        
        this.smallestBoundary = smallest;
        this.largestBoundary = largest;
    }
    
    public boolean isLeafNode() {
        return octants == null || octants.isEmpty();
    }
    
    /**
     * returns Octant under specific index
     * 
     * @param index index of the octant returned.
     *              Must be between 0 and 7 including.
     * @return Octant under specific index
     */
    public OctNode getOctant(int index) {
        if (!(0 <= index && index < 8)) {
            throw new IllegalArgumentException("getOctant passed with illegal index, can only be 0 to 7");
        }
        if (isLeafNode()) {
            return null;
        }
        return octants.get(index);
    }
    
    /**
     * Returns 3D location of vertices stored in this node
     * @return 3D location of vertices stored in this node
     */
    public MeshPoint getLocation() {
        return location;
    }
    
    /**
     * returns boundary box - corner with smallest coordinates.
     * @return boundary box - corner with smallest coordinates.
     */
    public Point3d getSmallBoundary() {
        return smallestBoundary;
    }

    /**
     * returns boundary box - corner with largest coordinates.
     * @return boundary box - corner with largest coordinates.
     */
    public Point3d getLargeBoundary() {
        return largestBoundary;
    }
    
    /**
     * Returns a map of all mesh facets that share the stored vertex. 
     * Value in the map contains the index which the 
     * vertex is stored in the mesh facet.
     * 
     * @return Map of facets sharing the stored mesh vertex
     */
    public Map<MeshFacet, Integer> getFacets() {
        return Collections.unmodifiableMap(facets);
    }
    
    /**
     * Set octants / children
     *
     * @param octants children of this node.
     *                Must not be null and has to have size of 8
     * 
     */
    protected void setOctants(List<OctNode> octants) {
        if (octants == null || octants.size() != 8) {
            throw new IllegalArgumentException("setOctants passed with null or illegal size");
        }
        this.location = null;
        this.octants = octants;
    }

    @Override
    public String toString() {
        String result = "[" + smallestBoundary + ", " + largestBoundary + "]";
        if (isLeafNode() && location != null) {
            result += " " + location.getPosition();
        }
        return result;
    }

    protected String toString(String prefix) {
        if (isLeafNode()) {
            return adjustPrefix(prefix) + toString() + '\n';
        }
        StringBuilder result = new StringBuilder(adjustPrefix(prefix));
        result.append(toString()).append('\n');
        if (prefix.endsWith("\\")) {
            prefix = prefix.substring(0, prefix.length() - 1) + ' ';
        }
        for (int i = 0; i < 7; i++) {
            result.append(getOctant(i).toString(prefix + '|'));
        }
        result.append(getOctant(7).toString(prefix + '\\'));
        return result.toString();
    }
    
    private String adjustPrefix(String prefix) {
        if (prefix.isEmpty()) {
            return "";
        }
        StringBuilder result = new StringBuilder();
        for (int i = 0; i < prefix.length() - 1; i++) {
            result.append(prefix.charAt(i)).append("   ");
        }
        result.append(prefix.charAt(prefix.length() - 1)).append("-- ");
        return result.toString();
    }
}
