package cz.fidentis.analyst.mesh.core;

import java.util.Objects;
import javax.vecmath.Point3d;
import javax.vecmath.Vector3d;

/**
 * MeshPoint represents a point with position, normal, and texture coordinates.
 *
 * @author Matej Lukes
 */
public class MeshPointImpl implements MeshPoint {
    
    private Point3d position;
    private Vector3d normal;
    private Vector3d texCoord;

    /**
     * constructor of meshPoint
     *
     * @param position position of MeshPoint
     * @param normal   normal of MeshPoint
     * @param texCoord coordinates in texture
     */
    public MeshPointImpl(Point3d position, Vector3d normal, Vector3d texCoord) {
        if (position == null) {
            throw new IllegalArgumentException("position cannot be null");
        } else {
            this.position = new Point3d(position);
        }
        
        if (normal != null) {
            this.normal = new Vector3d(normal);
        }
        
        if (texCoord != null) {
            this.texCoord = new Vector3d(texCoord);
        }        
    }

    /**
     * copy constructor of meshPoint
     * 
     * @param meshPoint copied meshPoint
     */
    public MeshPointImpl(MeshPoint meshPoint) {
        this(meshPoint.getPosition(), meshPoint.getNormal(), meshPoint.getTexCoord());
    }

    @Override
    public Vector3d getNormal() {
        return normal;
    }

    @Override
    public Point3d getPosition() {
        return position;
    }
    
    @Override
    public void setPosition(Point3d newPos) {
        if (newPos != null) {
            this.position = new Point3d(newPos);
        }
    }
    
    @Override
    public void setNormal(Vector3d newNormal) {
        this.normal = new Vector3d(newNormal);
    }

    @Override
    public Vector3d getTexCoord() {
        return texCoord;
    }
    
    @Override
    public boolean equals(Object obj) {
        if (!(obj instanceof MeshPointImpl)) {
            return false;
        }

        MeshPointImpl other = (MeshPointImpl) obj;
        
        if (!this.position.equals(other.position)) {
            return false;
        }
        
        if (this.normal == null) {
            if (other.normal != null) {
                return false;
            } 
        } else {
            if (other.normal == null) {
                return false;
            } 
            if (!this.normal.equals(other.normal)) {
                return false;
            }
        }
        
        if (this.texCoord == null) {
            if (other.texCoord != null) {
                return false;
            } 
        } else {
            if (other.texCoord == null) {
                return false;
            } 
            if (!this.texCoord.equals(other.texCoord)) {
                return false;
            }
        }
        
        return true;
    }

    @Override
    public int hashCode() {
        int hash = 5;
        hash = 67 * hash + Objects.hashCode(this.position);
        hash = 67 * hash + Objects.hashCode(this.normal);
        hash = 67 * hash + Objects.hashCode(this.texCoord);
        return hash;
    }

    @Override
    public String toString() {
        StringBuilder ret = new StringBuilder();
        ret.append("p=(").append(String.format("%.2f", position.x)).
                append("  ").append(String.format("%.2f", position.y)).
                append("  ").append(String.format("%.2f", position.z)).
                append(") ");
        if (normal != null) {
            ret.append("n=(").append(String.format("%.2f", normal.x)).
                    append("  ").append(String.format("%.2f", normal.y)).
                    append("  ").append(String.format("%.2f", normal.z)).
                    append(") ");
        }
        if (texCoord != null) {
            ret.append("t=(").append(String.format("%.2f", texCoord.x)).
                    append("  ").append(String.format("%.2f", texCoord.y)).
                    append("  ").append(String.format("%.2f", texCoord.z)).
                    append(") ");
        }
        return ret.toString();
    }
}
