package cz.fidentis.analyst.mesh.core;

import java.util.ArrayList;
import java.util.List;
import cz.fidentis.analyst.mesh.MeshVisitor;
import java.util.Collections;
import java.io.Serializable;
import java.util.Collection;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import javax.vecmath.Point3d;

/**
 * The main object for triangular meshes. Each mesh model consists  
 * of one or more mesh facets.
 * 
 * @author Matej Lukes
 * @author Radek Oslejsek
 */
public class MeshModel implements Serializable {
    
    private List<MeshFacet> facets = new ArrayList<>();
    
    /**
     * Constructor of MeshModel
     */
    public MeshModel() {

    }

    /**
     * Copy constructor of MeshModel
     *
     * @param meshModel copied MeshModel
     */
    public MeshModel(MeshModel meshModel) {
        for (MeshFacet facet: meshModel.facets) {
            facets.add(new MeshFacetImpl(facet));
        }
    }

    /**
     * change facets
     *
     * @param facets
     */
    public void setFacets(List<MeshFacet> facets) {
        this.facets = facets;
    }

    /**
     * returns list of MeshFacets
     *
     * @return list of MeshFacets
     */
    public List<MeshFacet> getFacets() {
        return Collections.unmodifiableList(facets);
    }

    /**
     * Adds a new mesh facet to the model. 
     *
     * @param facet new MeshFacet
     */
    public void addFacet(MeshFacet facet) {
        facets.add(facet);
    }
    
    /**
     * Removes facet from the model.
     * 
     * @param index Index of the facet
     * @return Removed facet or {@code null}
     * @throws IndexOutOfBoundsException if the facet does not exist
     */
    public MeshFacet removeFacet(int index) {
        return facets.remove(index);
    }
    
    /**
     * Adds a new mesh facets to the model. 
     *
     * @param newFacets collection of new new facets
     */
    public void addFacets(Collection<MeshFacet> newFacets) {
        facets.addAll(newFacets);
    }
    
    /**
     * Applies the visitor to all mesh facets. If the visitor is thread-safe
     * and the {@code concurrently} is {@code true}, then the visitor is
     * applied to all mesh facet concurrently using all CPU cores.
     * 
     * @param visitor Visitor to be applied for the computation
     * @param concurrently Parallel computation
     * @throws NullPointerException if the visitor is {@code null}
     */
    public void compute(MeshVisitor visitor, boolean concurrently) {
        if (concurrently && visitor.isThreadSafe()) {
            ExecutorService executor = Executors.newFixedThreadPool(Runtime.getRuntime().availableProcessors());
            
            for (MeshFacet f: this.facets) {
                Runnable worker = new Runnable() {
                    @Override
                    public void run() {
                        f.accept(visitor);
                    }
                };
                executor.execute(worker);
            }
            
            // Wait until all symmetry planes are computed:
            executor.shutdown();
            while (!executor.isTerminated()){}
        } else {
            for (MeshFacet f: this.facets) {
                f.accept(visitor);
            }
        }
    }
    
    /**
     * Applies the visitor to all mesh facets sequentially.
     * 
     * @param visitor Visitor to be applied for the computation
     * @throws NullPointerException if the visitor is {@code null}
     */
    public void compute(MeshVisitor visitor) {
        compute(visitor, false);
    }
    
    /**
     * Removes duplicate vertices that differ only in normal vectors or texture coordinates.
     * Multiple normals are replaced with the average normal. If the texture coordinate 
     * differ then randomly selected one is used.
     */
    public void simplifyModel() {
        for (MeshFacet f : this.facets) {
            f.simplify();
        }
    }
    
    @Override
    public String toString() {
        int verts = 0;
        for (MeshFacet f: facets) {
            verts += f.getNumberOfVertices();
        }
        return facets.size() + " facets with " + verts + " vertices";
    }
    
    /**
     * Returns number of vertices }sum of all facets).
     * @return Number of vertices
     */
    public long getNumVertices() {
        return facets
                .stream()
                .mapToInt(f -> f.getNumberOfVertices())
                .sum();
    }
    
    /**
     * Returns central point computed from mesh vertices
     * 
     * @return centroid
     */
    public Point3d getCentroid() {
        Point3d c = new Point3d(0, 0, 0);
        final long size = getNumVertices();
        getFacets().stream()
                .flatMap(f -> f.getVertices().parallelStream())
                .map(p -> p.getPosition())
                .forEach(p -> {
                    c.x += p.x / size;
                    c.y += p.y / size;
                    c.z += p.z / size;
                });
        return c;
    }
    
    
}
