package cz.fidentis.analyst.feature.services;

import cz.fidentis.analyst.feature.FeaturePoint;
import cz.fidentis.analyst.feature.provider.FeaturePointTypeProvider;
import cz.fidentis.analyst.feature.utils.FileResourcesUtils;
import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Stream;

/**
 * Class used to import feature points from file of CSV format
 * 
 * @author Jakub Kolman
 */
public class FeaturePointCsvLoader {
    
    private static final String PRIMARY_COLUMN_DELIMETER = ";";
    private static final String SECONDARY_COLUMN_DELIMETER = ",";
    private static final String CODE_PREFIX_DELIMETER = " ";

    /**
     * Loads feature points from file of csv format
     * @param path
     * @param fileName
     * @return
     * @throws FileNotFoundException
     * @throws IOException 
     */
    public static List<FeaturePoint> loadFeaturePoints(String path, String fileName) throws FileNotFoundException, IOException {
        FileResourcesUtils app = new FileResourcesUtils();
        try (InputStreamReader streamReader
                = new InputStreamReader(app.getFileAsStream(path, fileName), StandardCharsets.UTF_8);
                BufferedReader reader = new BufferedReader(streamReader)) {
            
            String headerLine = reader.readLine();
            final String delimiter;
            if (headerLine.contains(PRIMARY_COLUMN_DELIMETER)) {
                delimiter = PRIMARY_COLUMN_DELIMETER;
            } else if (headerLine.contains(SECONDARY_COLUMN_DELIMETER)) {
                delimiter = SECONDARY_COLUMN_DELIMETER;
            } else {
                throw new IOException(String.format("Feature point import file '%s' has wrong format - unknown delimiter", fileName));
            }

            Stream<String> lines = reader.lines();
            List<List<String>> linesList = new ArrayList<>();
            
            linesList.add(Arrays.asList(headerLine.split(delimiter, -1)));
            lines.forEach(line -> { 
                linesList.add(Arrays.asList(line.split(delimiter, -1)));
            });
            
            if (linesList.stream().anyMatch(list -> list.size() != linesList.get(0).size())) {
                throw new IOException(String.format("Feature point import file '%s' has wrong format", fileName));
            }

            // TODO: In real data sets, there can be multiple FP collections (lines) in a single file,
            // e.g., FPs before and after the face warping. Currently, we ignore them.
            List<FeaturePoint> points = new ArrayList<>();
            for (int i = 1; i < linesList.get(0).size(); i += 3) {
                if (linesList.get(1).get(i).isBlank() 
                        || linesList.get(1).get(i+1).isBlank() 
                        || linesList.get(1).get(i+2).isBlank()) { // skip missing points
                    continue;
                }
                FeaturePoint point = new FeaturePoint(
                        Double.parseDouble(linesList.get(1).get(i)),
                        Double.parseDouble(linesList.get(1).get(i + 1)),
                        Double.parseDouble(linesList.get(1).get(i + 2)),
                        FeaturePointTypeProvider.getInstance().getFeaturePointTypeByCode(
                                getCode(linesList.get(0).get(i))
                        )
                );
                points.add(point);
            }
            return points;

        } catch (NumberFormatException e1) {
            throw new IOException(e1);
        }
    }

    private static String getCode(String str) {
        return str.substring(0, str.indexOf(CODE_PREFIX_DELIMETER));
    }
}
