package cz.fidentis.analyst;

import cz.fidentis.analyst.face.HumanFace;
import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;


/**
 * This class encapsulates data for a "comparison project", i.e., a project 
 * enabling analysts to compare and investigate human faces.
 *
 * @author Matej Kovar
 */
public class Project {
    
    private boolean saved = true;

    private List<HumanFace> faces = new ArrayList<>();
    
    /* Project data (paths to faces, opened tabs..) */
    private ProjectConfiguration cfg = new ProjectConfiguration();
    
    /**
     * Asks whether project is saved or some change was made
     * @return true if project is saved, false otherwise
     */
    public boolean isSaved() {
        return saved;
    }

    public void setSaved(boolean saved) {
        this.saved = saved;
    }

    public ProjectConfiguration getCfg() {
        return cfg;
    }

    public void setCfg(ProjectConfiguration cfg) {
        this.cfg = cfg;
    }
    
    /**
     * Adds new path to project configuration
     * @param path Path to be added
     * @return true if path was successfully added
     */
    public boolean addNewPath(Path path) {
        return this.cfg.addPath(path);
    }
    
    /**
     * Checks whether path is already loaded in project configuration
     * @param path Path to be checked
     * @return true if path was loaded, false otherwise
     */
    public boolean pathLoaded(Path path) {
        return this.cfg.getPaths().contains(path);
    }
    
    /**
     * Adds new FaceToFace tab to project configuration
     * @param name1 String name of first face
     * @param name2 String name of second face
     */
    public void addNewFaceToFaceTabFace(String name1, String name2) {
        this.cfg.addFaceToFaceTabFace(name1, name2);
    }
    
    /**
     * Adds new face to FaceTab
     * @param name String name of face
     */
    public void addNewSingleFaceTabFace(String name) {
        this.cfg.addSingleFaceTabFace(name);
    }
    
    /**
     * Removes FaceTab
     * @param name String name of face
     */
    public void removeFaceTab(String name) {
        this.cfg.removeFaceTab(name);
    }
    
    /**
     * Removes FaceToFace tab
     * @param name1 String name of first face
     * @param name2 String name of second face
     */
    public void removeFaceToFaceTab(String name1, String name2) {
        this.cfg.removeFaceToFaceTab(name1, name2);
    }
    
    /**
     * Returns list of HumanFace secondary faces
     * 
     * @return list of secondary faces
     */
    public List<HumanFace> getFaces() {
        return Collections.unmodifiableList(faces);
    }

    /**
     * Sets new list of secondary faces
     * 
     * @param secondaryFaces list of HumanFace which will be new list of 
     * secondary faces
     * @throws IllegalArgumentException if one of faces from argument is null
     */
    public void setFaces(List<HumanFace> secondaryFaces) {
        this.faces.clear();
        for (int i = 0; i < secondaryFaces.size(); i++) {
            if (secondaryFaces.get(i) == null) {
                throw new IllegalArgumentException("One of faces is null");
            }
        }
        this.faces.addAll(secondaryFaces);
    }
    
    /**
     * Adds new face to faces
     * 
     * @param face HumanFace which will be added to list of secondary faces
     * @throws IllegalArgumentException when argument face is null
     */
    public void addFace(HumanFace face) {
        if (face == null) {
            throw new IllegalArgumentException("Face is null");
        }
        this.faces.add(face);
    }
    
    /**
     * Removes HumanFace from faces
     * 
     * @param face HumanFace which will be removed from faces
     * @throws IllegalArgumentException when argument face is null
     */
    public void removeFace(HumanFace face) {
        if (face == null) {
            throw new IllegalArgumentException("Face is null");
        }
        
        for (int i = 0; i < faces.size(); i++) {
            if (faces.get(i).equals(face)) {
                faces.remove(i);
            }
        } 
    }
    
    /**
     * Removes face by providing its name
     * 
     * @param name name of the face to be removed
     */
    public void removeFaceByName(String name) {
        HumanFace face = this.getFaceByName(name);
        if (face != null) {
            this.removeFace(face);
        }
        
        this.cfg.removePath(name);
        //this.cfg.removeFaceTab(name);
        
    }
    
    /**
     * Removes all faces from list of faces
     */
    public void removeAll() {
        faces.clear();
    }
    
    /**
     * Removes faces which are sent to this function by list of HumanFace
     * from faces
     * 
     * @param faces List of HumanFace faces which should be removed from
     * faces
     */
    public void removeSelected(List<HumanFace> faces) {
        for (int i = 0; i < faces.size(); i++) {
            this.removeFace(faces.get(i));
        }
    }
    
    
    /**
     * 
     * @param name String name of the model (face)
     * @return face from this project faces, if not in project than returns 
     * null
     */
    public HumanFace getFaceByName(String name) {
        for (HumanFace face : faces) {
            if (face.getShortName().equals(name)) {
                return face;
            }
        }
        return null;
    }
    
    /**
     * Loads face from path
     * @param name String name of face
     * @return loaded HumanFace
     */
    public HumanFace loadFace(String name) {
        HumanFace face = this.getFaceByName(name);
        if (face == null) {
            try {
                Logger out = Logger.measureTime();
                Path path = this.getCfg().getPathToFaceByName(name);
                File file = path.toFile();
                face = new HumanFace(file, true); // loads also landmarks, if exist
                
                String pathString = path.toString();
                Path preview = Paths.get(pathString.substring(0, pathString.lastIndexOf(".")).concat("_preview.jpg"));
                //Path previewSmall = Paths.get(pathString.substring(0, pathString.lastIndexOf(".")).concat("_preview_small.png"));
                face.setPreview(preview);
                this.addFace(face);
                out.printDuration("Loaded model " + face.getShortName() +" with " + face.getMeshModel().getNumVertices() + " vertices");
            } catch (IOException ex) {
                //ex.printStackTrace();
                Logger.print(ex.toString());
            }    
        }
        return face;
    }
   
}
