package cz.fidentis.analyst.distance;

import cz.fidentis.analyst.canvas.Canvas;
import cz.fidentis.analyst.core.LoadedActionEvent;
import cz.fidentis.analyst.core.ControlPanelAction;
import cz.fidentis.analyst.feature.FeaturePointType;
import cz.fidentis.analyst.scene.DrawableFeaturePoints;
import cz.fidentis.analyst.visitors.face.HausdorffDistancePrioritized;
import cz.fidentis.analyst.visitors.mesh.HausdorffDistance.Strategy;
import java.awt.Color;
import java.awt.event.ActionEvent;
import java.util.HashMap;
import java.util.Map;
import javax.swing.JComboBox;
import javax.swing.JTabbedPane;
import javax.swing.JTextField;
import javax.swing.JToggleButton;

/**
 * Action listener for the curvature computation.
 * 
 * @author Radek Oslejsek
 * @author Daniel Schramm
 */
public class DistanceAction extends ControlPanelAction {
    
    /*
     * Attributes handling the state
     */
    private HausdorffDistancePrioritized visitor = null;
    private final Map<FeaturePointType, Double> featurePointTypes = new HashMap<>();
    private String strategy = DistancePanel.STRATEGY_POINT_TO_POINT;
    private boolean relativeDist = false;
    
    private final DistancePanel controlPanel;
    
    /**
     * Constructor.
     * 
     * @param canvas OpenGL canvas
     * @param topControlPanel Top component for placing control panels
     */
    public DistanceAction(Canvas canvas, JTabbedPane topControlPanel) {
        super(canvas, topControlPanel);
        this.controlPanel = new DistancePanel(this, getSecondaryFeaturePoints().getFeaturePoints());
    }

    @Override
    public void actionPerformed(ActionEvent ae) {
        String action = ae.getActionCommand();
        
        switch (action) {
            case DistancePanel.ACTION_COMMAND_SHOW_HIDE_PANEL:
                hideShowPanelActionPerformed(ae, this.controlPanel);
                break;
            case DistancePanel.ACTION_COMMAND_SHOW_HIDE_HEATMAP:
                if (((JToggleButton) ae.getSource()).isSelected()) {
                    setHeatmap();
                    getSecondaryDrawableFace().setRenderHeatmap(true);
                } else {
                    getSecondaryDrawableFace().setRenderHeatmap(false);
                }
                break;
            case DistancePanel.ACTION_COMMAND_SET_DISTANCE_STRATEGY:
                strategy = (String) ((JComboBox) ae.getSource()).getSelectedItem();
                this.visitor = null; // recompute
                setHeatmap();
                break;
            case DistancePanel.ACTION_COMMAND_RELATIVE_ABSOLUTE_DIST:
                this.relativeDist = ((JToggleButton) ae.getSource()).isSelected();
                this.visitor = null; // recompute
                setHeatmap();
                break;
            case DistancePanel.ACTION_COMMAND_FEATURE_POINT_HIGHLIGHT:
                highlightFeaturePoint((LoadedActionEvent) ae);
                break;
            case DistancePanel.ACTION_COMMAND_FEATURE_POINT_RESIZE:
                resizeFeaturePoint((LoadedActionEvent) ae);
                break;
            case DistancePanel.ACTION_COMMAND_WEIGHTED_DIST_RECOMPUTE:
                this.visitor = null; // recompute
                setHeatmap();
                break;
            default:
                // to nothing
        }
        renderScene();
    }
    
    protected void setHeatmap() {
        Strategy useStrategy;
        switch (strategy) {
            case DistancePanel.STRATEGY_POINT_TO_POINT:
                useStrategy = Strategy.POINT_TO_POINT;
                break;
            case DistancePanel.STRATEGY_POINT_TO_TRIANGLE:
                useStrategy = Strategy.POINT_TO_TRIANGLE_APPROXIMATE;
                break;
            default:
                throw new UnsupportedOperationException(strategy);
        }
        
        if (visitor == null) {
            this.visitor = new HausdorffDistancePrioritized(getPrimaryDrawableFace().getModel(),
                    featurePointTypes,
                    useStrategy,
                    relativeDist,
                    true);
            getSecondaryDrawableFace().getHumanFace().accept(visitor);
        }
        
        getSecondaryDrawableFace().setHeatMap(visitor.getDistances());
    }

    private void highlightFeaturePoint(LoadedActionEvent actionEvent) {
        final int index = (int) actionEvent.getData();
        final FeaturePointType fpType = getTypeOfFeaturePoint(index);
        final DrawableFeaturePoints secondaryFeaturePoints = getSecondaryFeaturePoints();
        
        if (((JToggleButton) actionEvent.getSource()).isSelected()) {
            secondaryFeaturePoints.setColor(index, Color.MAGENTA);
            featurePointTypes.put(fpType, secondaryFeaturePoints.getSize(index));
        } else {
            secondaryFeaturePoints.resetColorToDefault(index);
            featurePointTypes.remove(fpType);
        }
    }

    private void resizeFeaturePoint(LoadedActionEvent actionEvent) {
        final int index = (int) actionEvent.getData();
        final double size = Double.parseDouble(((JTextField) actionEvent.getSource()).getText());
        
        getSecondaryFeaturePoints().setSize(index, size);
        featurePointTypes.replace(getTypeOfFeaturePoint(index), size);
    }
    
    private FeaturePointType getTypeOfFeaturePoint(int index) {
        return getSecondaryFeaturePoints().getFeaturePoints()
                .get(index)
                .getFeaturePointType();
    }
}
