package cz.fidentis.analyst.visitors.kdtree;

import cz.fidentis.analyst.kdtree.KdTree;
import cz.fidentis.analyst.kdtree.KdTreeVisitor;
import cz.fidentis.analyst.visitors.Distance;
import javax.vecmath.Vector3d;

/**
 * This visitor finds the minimal distance between the given 3D point and points 
 * (e.g., mesh vertices) stored in k-d trees. 
 * Only distance is computed and returned without the corresponding closest point locations.
 * <p>
 * This visitor is thread-safe, i.e., a single instance of the visitor can be used 
 * to inspect multiple k-d trees simultaneously. 
 * </p>
 * 
 * @author Daniel Schramm
 * @author Radek Oslejsek
 */
public class KdTreeDistance extends KdTreeVisitor implements Distance {
    
    private double distance = Double.POSITIVE_INFINITY;
    private Vector3d point3d;
    
    /**
     * Constructor.
     * @param point A 3D point from which distance is computed. Must not be {@code null}
     * @throws {@code IllegalArgumentException} if some parameter is wrong
     */
    public KdTreeDistance(Vector3d point) {
        if (point == null) {
            throw new IllegalArgumentException("point");
        }
        this.point3d = point;
    }

    @Override
    public double getDistance() {
        return distance;
    }

    @Override
    public void visitKdTree(KdTree kdTree) {
        final KdTreeClosestNode visitor = new KdTreeClosestNode(point3d);
        kdTree.accept(visitor);
        
        synchronized (this) {
            if (visitor.getDistance() < distance) {
                distance = visitor.getDistance();
            }
        }
    }

}
