package cz.fidentis.analyst.visitors.face;

import cz.fidentis.analyst.face.HumanFace;

/**
 * A functor. When instantiated, it can be gradually applied to multiple faces.
 * It inspects the state of the human face one by one, and (cumulatively) computes results.
 * <p>
 * Implement this interface whenever you want to define new algorithm over a human face.
 * </p>
 * <p>
 * If the visitor is thread-safe, then a single instance of the visitor
 * can visit concurrently (and asynchronously) multiple faces. Otherwise, 
 * the parallel inspection is still possible, but a new instance of the visitor 
 * has to be used for each human face.
 * </p>
 *
 * @author Daniel Schramm
 */
public abstract class HumanFaceVisitor {
    
    /**
     * Returns {@code true} if the implementation is thread-safe and then
     * <b>a single visitor instance</b> can be applied to multiple faces simultaneously.
     * <p>
     * Thread-safe implementation means that any read or write from/to the visitor's 
     * state is protected by {@code synchronized}.
     * </p>
     * 
     * @return {@code true} if the implementation is thread-safe.
     */
    public boolean isThreadSafe() {
        return true;
    }
    
    /**
     * The main inspection method to be implemented by specific visitors.
     * 
     * @param humanFace Face to be visited
     */
    public abstract void visitHumanFace(HumanFace humanFace);
}
