package cz.fidentis.analyst.scene;

import cz.fidentis.analyst.face.HumanFace;
import java.awt.Color;
import java.util.ArrayList;
import java.util.List;

/**
 * A simple 3D scene. The structure is as follows:
 * Drawable components of a single human face (mesh, symmetry plane, feature points. etc.)
 * are always stored at the same index. One face can be labeled as "primary", one as "secondary".
 * Other faces have no label.
 * Then, there is a special list of so-called other drawables. The size of this list can differ.
 * 
 * @author Radek Oslejsek
 * @author Dominik Racek
 */
public class Scene {
    
    private final List<DrawableFace> drawableFaces = new ArrayList<>();
    private final List<DrawableFeaturePoints> drawableFeaturePoints = new ArrayList<>();
    private final List<DrawablePlane> drawableSymmetryPlanes = new ArrayList<>();
    private final List<Drawable> otherDrawables = new ArrayList<>();
    
    private int primaryFaceIndex = -1;
    private int secondaryFaceIndex = -1;
    
    public static final int MAX_FACES_IN_SCENE = 20;
    
    /**
     * Removes all objects.
     */
    public void clearScene() {
        this.drawableFaces.clear();
        this.drawableFeaturePoints.clear();
        this.drawableSymmetryPlanes.clear();
        this.otherDrawables.clear();
        primaryFaceIndex = -1;
        secondaryFaceIndex = -1;
    }
    
    /**
     * Finds and returns a first free index for human face and its drawables.
     * @return a first free index for human face and its drawables.
     */
    public int getFreeIndex() {
        int index = -1;
        for (int i = 0; i < getArraySize(); i++) {
            if (this.drawableFaces.get(i) == null) {
                index = i;
                break;
            }
        }
        if (index == -1) {
            index = getArraySize();
        }
        return (index < Scene.MAX_FACES_IN_SCENE) ? index : -1;
    }
    
    /**
     * Finds and returns a first free index for other drawable objects.
     * @return a first free index for other drawable objects.
     */
    public int getFreeIndexForOtherDrawables() {
        int index = -1;
        for (int i = 0; i < this.otherDrawables.size(); i++) {
            if (this.otherDrawables.get(i) == null) {
                index = i;
                break;
            }
        }
        if (index == -1) {
            index = otherDrawables.size();
        }
        return (index < Scene.MAX_FACES_IN_SCENE) ? index : -1;
    }
    
    /**
     * Returns index of the primary face or -1
     * @return index of the primary face or -1
     */
    public int getPrimaryFaceIndex() {
        return this.primaryFaceIndex;
    }
    
    /**
     * Returns index of the secondary face or -1
     * @return index of the secondary face or -1
     */
    public int getSecondaryFaceIndex() {
        return this.secondaryFaceIndex;
    }
    
    /**
     * Sets the given face as primary.
     * @param index Index of the face that should be set as primary
     */
    public void setFaceAsPrimary(int index) {
        if (index >= 0 && index < getArraySize()) {
            this.primaryFaceIndex = index;
        }
    }
    
    /**
     * Sets the given face as secondary.
     * @param index Index of the face that should be set as secondary
     */
    public void setFaceAsSecondary(int index) {
        if (index >= 0 && index < getArraySize()) {
            this.secondaryFaceIndex = index;
        }
    }
    
    /**
     * Returns all "occupied" indices, i.e., indexes where is stored some human face.
     * @return "occupied" indices, i.e., indexes where is stored some human face.
     */
    public List<Integer> getFaceIndices() {
        List<Integer> ret = new ArrayList<>();
        for (int i = 0; i < this.drawableFaces.size(); i++) {
            if (this.drawableFaces.get(i) != null) {
                ret.add(i);
            }
        }
        return ret;
    }
    
    /**
     * Returns drawable face.
     * 
     * @param index Index of the face
     * @return drawable face or {@code null}
     */
    public DrawableFace getDrawableFace(int index) {
        return (index < 0 || index >= getArraySize()) ? null : drawableFaces.get(index);
    }
    
    /**
     * Sets the face and all its existing drawable components.   
     * If the face is {@code null}, then the drawable face and all its components are removed.
     * 
     * @param index Index of the face
     * @param face New face or {@code null}
     * @return 
     */
    public boolean setHumanFace(int index, HumanFace face) {
        return setDrawableFace(index, face) &&
                setDrawableFeaturePoints(index, face) &&
                setDrawableSymmetryPlane(index, face);
    }
    
    /**
     * Sets the drawable face (mesh). If the face is {@code null}, then the drawable face is removed.
     *
     * @param index Index of the face
     * @param face New face or {@code null}
     */
    public boolean setDrawableFace(int index, HumanFace face) {
        if (index < 0 || index >= Scene.MAX_FACES_IN_SCENE) {
            return false;
        } else if (face == null) { // remove
            if (index >= getArraySize()){
                return false;
            } else {
                this.drawableFaces.set(index, null);
                return true;
            }
        } else if (!prepareArrays(index, face)) {
            return false;
        }        
        
        drawableFaces.set(index, new DrawableFace(face));
        
        return true;
    }
    
    /**
     * Sets the drawable feature points of the face. 
     * If the face is {@code null}, then the drawable feature points are removed.
     *
     * @param index Index of the face
     * @param face New face or {@code null}
     */
    public boolean setDrawableFeaturePoints(int index, HumanFace face) {
        if (index < 0 || index >= Scene.MAX_FACES_IN_SCENE) {
            return false;
        } else if (face == null) { // remove
            if (index >= getArraySize()){
                return false;
            } else {
                this.drawableFeaturePoints.set(index, null);
                return true;
            }
        } else if (!prepareArrays(index, face)) {
            return false;
        }        
        
        drawableFeaturePoints.set(
                index,
                (face.getFeaturePoints() != null) 
                        ? new DrawableFeaturePoints(face.getFeaturePoints())
                        : null
        );
        
        return true;
    }
    
    /**
     * Sets the drawable symmetry plane of the face. 
     * If the face is {@code null}, then the drawable symmetry plane is removed.
     *
     * @param index Index of the face
     * @param face New face or {@code null}
     */
    public boolean setDrawableSymmetryPlane(int index, HumanFace face) {
        if (index < 0 || index >= Scene.MAX_FACES_IN_SCENE) {
            return false;
        } else if (face == null) { // remove
            if (index >= getArraySize()){
                return false;
            } else {
                this.drawableSymmetryPlanes.set(index, null);
                return true;
            }
        } else if (!prepareArrays(index, face)) {
            return false;
        }        
        
        drawableSymmetryPlanes.set(
                index,
                (face.getSymmetryPlane() != null && face.getSymmetryPlaneFacet() != null) 
                        ? new DrawablePlane(face.getSymmetryPlaneFacet(), face.getSymmetryPlane())
                        : null
        );
        
        return true;
    }
    
    /**
     * Sets other drawable. 
     * If the drawable is {@code null}, then the drawable is removed.
     *
     * @param index Index of the drawable
     * @param face New face or {@code null}
     */
    public boolean setOtherDrawable(int index, Drawable dr) {
        if (index < 0 || index >= Scene.MAX_FACES_IN_SCENE) {
            return false;
        } else if (dr == null) { // remove
            if (index >= otherDrawables.size()){
                return false;
            } else {
                otherDrawables.set(index, null);
                return true;
            }
        } 
        
        for (int i = otherDrawables.size(); i <= index; i++) {
            otherDrawables.add(null);
        }
        otherDrawables.set(index, dr);
        return true;
    }
    
    /**
     * Sets default colors of faces and feature points
     */
    public final void setDefaultColors() {
        for (int i = 0; i < getArraySize(); i++) {
            if (drawableFaces.get(i) != null) {
                if (i == this.primaryFaceIndex) {
                    drawableFaces.get(i).setColor(DrawableFace.SKIN_COLOR_PRIMARY);
                } else if (i == this.secondaryFaceIndex) {
                    drawableFaces.get(i).setColor(DrawableFace.SKIN_COLOR_SECONDARY);
                } else {
                    drawableFaces.get(i).setColor(DrawableFace.SKIN_COLOR_DEFAULT);
                }
            }
            if (drawableFeaturePoints.get(i) != null) {
                if (i == this.primaryFaceIndex) {
                    drawableFeaturePoints.get(i).setColor(getColorOfFeaturePoints(DrawableFace.SKIN_COLOR_PRIMARY));
                } else if (i == this.secondaryFaceIndex) {
                    drawableFeaturePoints.get(i).setColor(getColorOfFeaturePoints(DrawableFace.SKIN_COLOR_SECONDARY));
                } else {
                    drawableFeaturePoints.get(i).setColor(getColorOfFeaturePoints(DrawableFace.SKIN_COLOR_DEFAULT));
                }
            }
            /*
            if (drawableSymmetryPlanes.get(i) != null) {
                drawableSymmetryPlanes.get(i).setColor(
                        (i == 0) ? DrawableFace.SKIN_COLOR_PRIMARY.darker() : DrawableFace.SKIN_COLOR_SECONDARY.darker()
                ); 
            }
            */
        }
    }

    /**
     * Returns drawable feature points.
     * 
     * @param index Index of the face
     * @return drawable face or {@code null}
     */
    public DrawableFeaturePoints getDrawableFeaturePoints(int index) {
        return (index >= 0 && index < getArraySize()) ? drawableFeaturePoints.get(index) : null;
    }
    
    /**
     * Returns drawable symmetry plane.
     * 
     * @param index Index of the face
     * @return drawable plane or {@code null}
     */
    public DrawablePlane getDrawableSymmetryPlane(int index) {
        return (index >= 0 && index < getArraySize()) ? drawableSymmetryPlanes.get(index) : null;
    }
    
    /**
     * Showing or hiding the face (its mesh)
     *
     * @param index Index of the face
     * @param show determines whether to hide or show the object
     */
    public void showDrawableFace(int index, boolean show) {
        if (index >= 0 && index < getArraySize() && this.drawableFaces.get(index) != null) {
            this.drawableFaces.get(index).show(show);
        }
    }

    /**
     * Showing or hiding the feature points
     *
     * @param index Index of the face
     * @param show determines whether to hide or show the object
     */
    public void showFeaturePoints(int index, boolean show) {
        if (index >= 0 && index < getArraySize() && this.drawableFeaturePoints.get(index) != null) {
            this.drawableFeaturePoints.get(index).show(show);
        }
    }
    
    /**
     * Showing or hiding the symmetry plane
     *
     * @param index Index of the face
     * @param show determines whether to hide or show the object
     */
    public void showSymmetryPlane(int index, boolean show) {
        if (index >= 0 && index < getArraySize() && this.drawableSymmetryPlanes.get(index) != null) {
            this.drawableSymmetryPlanes.get(index).show(show);
        }
    }

    /**
     * Returns other drawable object.
     * 
     * @return drawable object or {@code null}
     */
    public Drawable getOtherDrawable(int index) {
        return (index >= 0 && index < this.otherDrawables.size()) ? this.otherDrawables.get(index) : null;
    }
    
    /**
     * Returns all drawable objects.
     * 
     * @return all drawable objects.
     */
    public List<Drawable> getAllDrawables() {
        List<Drawable> ret = new ArrayList<>();
        ret.addAll(this.drawableFaces);
        ret.addAll(this.drawableFeaturePoints);
        ret.addAll(this.drawableSymmetryPlanes);
        ret.addAll(this.otherDrawables);
        while (ret.remove(null)) {}
        return ret;
    }
    
    protected Color getColorOfFeaturePoints(Color origColor) {
        return new Color(
                Math.min(255, origColor.getRed() + 40),
                Math.min(255, origColor.getGreen() + 40),
                Math.min(255, origColor.getBlue() + 40)
        );
    }
    
    protected boolean prepareArrays(int index, Object drawable) {
        if (index >= Scene.MAX_FACES_IN_SCENE) {
            return false;
        }
        
        // extend the 
        for (int i = getArraySize(); i <= index; i++) {
            this.drawableFaces.add(null);
            this.drawableFeaturePoints.add(null);
            this.drawableSymmetryPlanes.add(null);
            //this.drawableCuttingPlanes.add(null);
        }
        
        return true;
    }

    protected int getArraySize() {
        return this.drawableFaces.size();
    }
    
    @Override
    public String toString() {
        return this.drawableFaces.toString();
    }
}
