package cz.fidentis.analyst.batch;

import cz.fidentis.analyst.Logger;
import cz.fidentis.analyst.core.ProgressDialog;
import cz.fidentis.analyst.face.HumanFace;
import cz.fidentis.analyst.face.HumanFaceFactory;
import cz.fidentis.analyst.visitors.mesh.HausdorffDistance;
import java.nio.file.Path;
import java.util.List;

/**
 * A task that computes similarity of a set of faces by applying two-way 
 * Hausdorff distance to all pairs if the set.
 * The exact computation parameters are taken from the {@code BatchPanel}.
 * 
 * @author Radek Oslejsek
 */
public class CompleteHausdorffDistTask extends SimilarityTask {

    private final Stopwatch totalTime = new Stopwatch("Total computation time:\t");
    private final Stopwatch hdComputationTime = new Stopwatch("Hausdorff distance time:\t");
    private final Stopwatch loadTime = new Stopwatch("Disk access time:\t");
    
    /**
     * Constructor.
     * 
     * @param progressDialog A window that show the progress of the computation. Must not be {@code null}
     * @param controlPanel A control panel with computation parameters. Must not be {@code null}
     */
    public CompleteHausdorffDistTask(ProgressDialog progressDialog, BatchPanel controlPanel) {
        super(progressDialog, controlPanel);
    }

    @Override
    protected Void doInBackground() throws Exception {
        HumanFaceFactory factory = getControlPanel().getHumanFaceFactory();
        List<Path> faces = getControlPanel().getFacePaths();

        factory.setReuseDumpFile(true); // it's safe because no changes are made to models 
        factory.setStrategy(HumanFaceFactory.Strategy.MRU);
        
        totalTime.start();
        
        int counter = 0;
        for (int i = 0; i < faces.size(); i++) {
            String priFaceId = factory.loadFace(faces.get(i).toFile());
            
            for (int j = i; j < faces.size(); j++) { // starts with "i"!
                
                if (isCancelled()) { // the user canceled the process
                    return null;
                }

                // (Re-)load the primary face (it could be dumped in the meantime) and load the second face
                loadTime.start();
                HumanFace priFace = factory.getFace(priFaceId); 
                String secFaceId = factory.loadFace(faces.get(j).toFile());
                HumanFace secFace = factory.getFace(secFaceId);
                loadTime.stop();
                
                //Logger.print(priFace.getShortName() + " - " + secFace.getShortName());
                
                // compute Huasdorff distance in both ways
                hdComputationTime.start();
                //priFace.computeKdTree(true);
                HausdorffDistance hd = new HausdorffDistance(
                        priFace.getMeshModel(), 
                        HausdorffDistance.Strategy.POINT_TO_POINT, 
                        false, // relative
                        true,  // parallel
                        false  // crop
                );
                secFace.getMeshModel().compute(hd, true);
                setSimilarity(j, i, hd.getStats().getAverage());
                
                //secFace.computeKdTree(true);
                hd = new HausdorffDistance(
                        secFace.getMeshModel(), 
                        HausdorffDistance.Strategy.POINT_TO_POINT, 
                        false, // relative
                        true,  // parallel
                        false  // crop
                ); 
                priFace.getMeshModel().compute(hd, true);
                setSimilarity(i, j, hd.getStats().getAverage());
                hdComputationTime.stop();
                
                // update progress bar
                int progress = (int) Math.round(100.0 * ++counter / numCycles(faces.size()));
                getProgressDialog().setValue(progress);
            }
        }
        
        totalTime.stop();

        printTimeStats();

        return null;
    }
    
    protected void printTimeStats() {
        Logger.print(hdComputationTime.toString());
        Logger.print(loadTime.toString());
        Logger.print(totalTime.toString());
    }
    
    protected double numCycles(int nFaces) {
        return 0.5 * nFaces * (nFaces + 1);
    }
}
