package cz.fidentis.analyst.batch;

import cz.fidentis.analyst.Logger;
import cz.fidentis.analyst.core.ProgressDialog;
import cz.fidentis.analyst.face.HumanFace;
import cz.fidentis.analyst.face.HumanFaceFactory;
import cz.fidentis.analyst.visitors.mesh.HausdorffDistance;
import java.nio.file.Path;
import java.util.List;

/**
 * A task that computes similarity of a set of faces by computing
 * the distance of faces to an average face and then combining these values
 * to get mutual similarity for all pairs.
 * The exact computation parameters are taken from the {@code BatchPanel}.
 * 
 * @author Radek Oslejsek
 */
public class ApproxHausdorffDistTask extends SimilarityTask {
    
    private final Stopwatch totalTime = new Stopwatch("Total computation time:\t");
    private final Stopwatch hdComputationTime = new Stopwatch("Hausdorff distance time:\t");
    private final Stopwatch loadTime = new Stopwatch("Disk access time:\t");
    
    //private HumanFace avgFace;
    private int templateFaceIndex;
    
    /**
     * Constructor.
     * 
     * @param progressDialog A window that show the progress of the computation. Must not be {@code null}
     * @param controlPanel A control panel with computation parameters. Must not be {@code null}
     * @param avgFace average face
     */
    public ApproxHausdorffDistTask(ProgressDialog progressDialog, BatchPanel controlPanel, int templateFaceIndex) {
        super(progressDialog, controlPanel);
        this.templateFaceIndex = templateFaceIndex;
    }
    
    @Override
    protected Void doInBackground() throws Exception {
        HumanFaceFactory factory = getControlPanel().getHumanFaceFactory();
        List<Path> faces = getControlPanel().getFacePaths();
        
        totalTime.start();
        
        factory.setReuseDumpFile(true); // it's safe because no changes are made to models 
        factory.setStrategy(HumanFaceFactory.Strategy.MRU); // keep first X faces in the memory
        
        // We don't need to reaload the initFace periodically for two reasons:
        //   - It is never dumped from memory to disk because we use MRU
        //   - Even if dumped, the face keeps in the mempry until we hold the pointer to it
        loadTime.start();
        String templateFaceId = factory.loadFace(faces.get(templateFaceIndex).toFile());
        HumanFace templateFace = factory.getFace(templateFaceId);
        loadTime.stop();
        
        
        double[] dist = new double[faces.size()];
        
        for (int i = 0; i < faces.size(); i++) {
            
            if (isCancelled()) { // the user canceled the process
                return null;
            }
            
            if (i != templateFaceIndex) {
                
                loadTime.start();
                String faceId = factory.loadFace(faces.get(i).toFile());
                HumanFace face = factory.getFace(faceId);
                loadTime.stop();

                hdComputationTime.start();
                templateFace.computeKdTree(false);
                HausdorffDistance hd = new HausdorffDistance(
                        templateFace.getKdTree(),
                        HausdorffDistance.Strategy.POINT_TO_POINT,
                        true, // relative
                        true, // parallel
                        true  // crop
                );
                face.getMeshModel().compute(hd, true);
                dist[i] = hd.getStats().getAverage();
                hdComputationTime.stop();
            }
            
            // update progress bar
            int progress = (int) Math.round(100.0 * (i+1) / faces.size());
            getProgressDialog().setValue(progress);
            
            //Logger.print(factory.toString());
        }
        
        hdComputationTime.start();
        for (int i = 0; i < faces.size(); i++) {
            for (int j = i; j < faces.size(); j++) {
                double d = Math.abs(dist[i] - dist[j]);
                setSimilarity(i, j, d);
                setSimilarity(j, i, d);
            }
        }
        hdComputationTime.stop();
        
        totalTime.stop();

        printTimeStats();

        return null;
    }
    
    protected void printTimeStats() {
        Logger.print(hdComputationTime.toString());
        Logger.print(loadTime.toString());
        Logger.print(totalTime.toString());
    }
    
    protected double numCycles(int nFaces) {
        return 0.5 * nFaces * (nFaces + 1);
    }
}
