package cz.fidentis.analyst.scene;

import cz.fidentis.analyst.face.HumanFace;
import java.awt.Color;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * A simple 3D scene. The structure is as follows:
 * Drawable components of a single human face (mesh, symmetry plane, feature points. etc.)
 * are always stored at the same index (slot). One face can be labeled as "primary", one as "secondary".
 * Other faces have no label.
 * Then, there is a special list of so-called other drawables. The size of this list can differ.
 * 
 * @author Radek Oslejsek
 * @author Dominik Racek
 */
public class Scene {
    
    private final List<DrawableFace> drawableFaces = new ArrayList<>();
    private final List<DrawableFeaturePoints> drawableFeaturePoints = new ArrayList<>();
    private final List<DrawablePlane> drawableSymmetryPlanes = new ArrayList<>();
    private final List<Drawable> otherDrawables = new ArrayList<>();
    
    private int primaryFaceSlot = -1;
    private int secondaryFaceSlot = -1;
    
    public static final int MAX_FACES_IN_SCENE = 20;
    
    /**
     * Removes all objects.
     */
    public void clearScene() {
        this.drawableFaces.clear();
        this.drawableFeaturePoints.clear();
        this.drawableSymmetryPlanes.clear();
        this.otherDrawables.clear();
        primaryFaceSlot = -1;
        secondaryFaceSlot = -1;
    }
    
    /**
     * Finds and returns a first free slot <b>for human face and its feature points and symmetry plane</b>.
     * Use {@link #getFreeSlotForOtherDrawables()} to get free slot of other objects.
     * 
     * @return a first free slot for human face and its feature points and symmetry plane.
     */
    public int getFreeSlotForFace() {
        int slot = -1;
        for (int i = 0; i < getArraySize(); i++) {
            if (this.drawableFaces.get(i) == null) {
                slot = i;
                break;
            }
        }
        if (slot == -1) {
            slot = getArraySize();
        }
        return (slot < Scene.MAX_FACES_IN_SCENE) ? slot : -1;
    }
    
    /**
     * Finds and returns a first free slot for other drawable objects.
     * @return a first free slot for other drawable objects.
     */
    public synchronized int getFreeSlotForOtherDrawables() {
        int slot = -1;
        for (int i = 0; i < this.otherDrawables.size(); i++) {
            if (this.otherDrawables.get(i) == null) {
                slot = i;
                break;
            }
        }
        if (slot == -1) {
            slot = otherDrawables.size();
        }
        return slot;
    }
    
    /**
     * Returns slot of the primary face or -1
     * @return slot of the primary face or -1
     */
    public int getPrimaryFaceSlot() {
        return this.primaryFaceSlot;
    }
    
    /**
     * Returns slot of the secondary face or -1
     * @return slot of the secondary face or -1
     */
    public int getSecondaryFaceSlot() {
        return this.secondaryFaceSlot;
    }
    
    /**
     * Sets the given face as primary.
     * @param slot slot of the face that should be set as primary
     */
    public void setFaceAsPrimary(int slot) {
        if (slot >= 0 && slot < getArraySize()) {
            this.primaryFaceSlot = slot;
        }
    }
    
    /**
     * Sets the given face as secondary.
     * @param slot slot of the face that should be set as secondary
     */
    public void setFaceAsSecondary(int slot) {
        if (slot >= 0 && slot < getArraySize()) {
            this.secondaryFaceSlot = slot;
        }
    }
    
    /**
     * Returns all "occupied" slots, i.e., indexes where some human face is stored.
     * @return "occupied" slots, i.e., indexes where some human face is stored.
     */
    public List<Integer> getFaceSlots() {
        List<Integer> ret = new ArrayList<>();
        for (int i = 0; i < this.drawableFaces.size(); i++) {
            if (this.drawableFaces.get(i) != null) {
                ret.add(i);
            }
        }
        return ret;
    }
    
    /**
     * Returns drawable face.
     * 
     * @param slot Slot of the face
     * @return drawable face or {@code null}
     */
    public DrawableFace getDrawableFace(int slot) {
        return (slot < 0 || slot >= getArraySize()) ? null : drawableFaces.get(slot);
    }
    
    /**
     * Sets the face and all its existing drawable components.   
     * If the face is {@code null}, then the drawable face and all its components are removed.
     * 
     * @param slot Slot of the face
     * @param face New face or {@code null}
     * @return 
     */
    public boolean setHumanFace(int slot, HumanFace face) {
        return setDrawableFace(slot, face) &&
                setDrawableFeaturePoints(slot, face) &&
                setDrawableSymmetryPlane(slot, face);
    }
    
    /**
     * Sets the drawable face (mesh). If the face is {@code null}, then the drawable face is removed.
     *
     * @param slot Slot of the face
     * @param face New face or {@code null}
     */
    public boolean setDrawableFace(int slot, HumanFace face) {
        if (slot < 0 || slot >= Scene.MAX_FACES_IN_SCENE) {
            return false;
        } else if (face == null) { // remove
            if (slot >= getArraySize()){
                return false;
            } else {
                this.drawableFaces.set(slot, null);
                return true;
            }
        } else if (!prepareArrays(slot, face)) {
            return false;
        }        
        
        drawableFaces.set(slot, new DrawableFace(face));
        return true;
    }
    
    /**
     * Sets the drawable feature points of the face. 
     * If the face is {@code null}, then the drawable feature points are removed.
     *
     * @param slot Slot of the face
     * @param face New face or {@code null}
     */
    public boolean setDrawableFeaturePoints(int slot, HumanFace face) {
        if (slot < 0 || slot >= Scene.MAX_FACES_IN_SCENE) {
            return false;
        } else if (face == null || face.getFeaturePoints() == null) { // remove
            if (slot >= getArraySize()){
                return false;
            } else {
                this.drawableFeaturePoints.set(slot, null);
                return true;
            }
        } else if (!prepareArrays(slot, face)) {
            return false;
        }        
        
        drawableFeaturePoints.set(slot, new DrawableFeaturePoints(face.getFeaturePoints()));
        return true;
    }
    
    /**
     * Sets the drawable symmetry plane of the face. 
     * If the face is {@code null}, then the drawable symmetry plane is removed.
     *
     * @param slot Slot of the face
     * @param face New face or {@code null}
     */
    public boolean setDrawableSymmetryPlane(int slot, HumanFace face) {
        if (slot < 0 || slot >= Scene.MAX_FACES_IN_SCENE) {
            return false;
        } else if (face == null || face.getSymmetryPlane() == null) { // remove
            if (slot >= getArraySize()){
                return false;
            } else {
                this.drawableSymmetryPlanes.set(slot, null);
                return true;
            }
        } else if (!prepareArrays(slot, face)) {
            return false;
        }
        
        if (drawableSymmetryPlanes.get(slot) == null) { // add new 
            drawableSymmetryPlanes.set(slot, new DrawablePlane(face.getSymmetryPlane(), face.getBoundingBox(), false));
        } else { // replace existing
            drawableSymmetryPlanes.get(slot).setPlane(face.getSymmetryPlane());
        }
        
        return true;
    }
    
    /**
     * Sets other drawable. 
     * If the drawable is {@code null}, then the drawable is removed.
     *
     * @param slot Slot of the drawable
     * @param face New face or {@code null}
     */
    public boolean setOtherDrawable(int slot, Drawable dr) {
        if (slot < 0 || slot >= Scene.MAX_FACES_IN_SCENE) {
            return false;
        } else if (dr == null) { // remove
            if (slot >= otherDrawables.size()){
                return false;
            } else {
                otherDrawables.set(slot, null);
                return true;
            }
        } 
        
        for (int i = otherDrawables.size(); i <= slot; i++) {
            otherDrawables.add(null);
        }
        otherDrawables.set(slot, dr);
        return true;
    }
    
    /**
     * Sets default colors of faces and feature points
     */
    public final void setDefaultColors() {
        for (int i = 0; i < getArraySize(); i++) {
            if (drawableFaces.get(i) != null) {
                if (i == this.primaryFaceSlot) {
                    drawableFaces.get(i).setColor(DrawableFace.SKIN_COLOR_PRIMARY);
                } else if (i == this.secondaryFaceSlot) {
                    drawableFaces.get(i).setColor(DrawableFace.SKIN_COLOR_SECONDARY);
                } else {
                    drawableFaces.get(i).setColor(DrawableFace.SKIN_COLOR_DEFAULT);
                }
            }
            if (drawableFeaturePoints.get(i) != null) {
                if (i == this.primaryFaceSlot) {
                    drawableFeaturePoints.get(i).setColor(getColorOfFeaturePoints(DrawableFace.SKIN_COLOR_PRIMARY));
                } else if (i == this.secondaryFaceSlot) {
                    drawableFeaturePoints.get(i).setColor(getColorOfFeaturePoints(DrawableFace.SKIN_COLOR_SECONDARY));
                } else {
                    drawableFeaturePoints.get(i).setColor(getColorOfFeaturePoints(DrawableFace.SKIN_COLOR_DEFAULT));
                }
            }
            /*
            if (drawableSymmetryPlanes.get(i) != null) {
                drawableSymmetryPlanes.get(i).setColor(
                        (i == 0) ? DrawableFace.SKIN_COLOR_PRIMARY.darker() : DrawableFace.SKIN_COLOR_SECONDARY.darker()
                ); 
            }
            */
        }
    }

    /**
     * Returns drawable feature points.
     * 
     * @param slot Slot of the face
     * @return drawable face or {@code null}
     */
    public DrawableFeaturePoints getDrawableFeaturePoints(int slot) {
        return (slot >= 0 && slot < getArraySize()) ? drawableFeaturePoints.get(slot) : null;
    }
    
    /**
     * Returns drawable symmetry plane.
     * 
     * @param slot Slot of the face
     * @return drawable plane or {@code null}
     */
    public DrawablePlane getDrawableSymmetryPlane(int slot) {
        return (slot >= 0 && slot < getArraySize()) ? drawableSymmetryPlanes.get(slot) : null;
    }
    
    /**
     * Showing or hiding the face (its mesh)
     *
     * @param slot Slot of the face
     * @param show determines whether to hide or show the object
     */
    public void showDrawableFace(int slot, boolean show) {
        if (slot >= 0 && slot < getArraySize() && this.drawableFaces.get(slot) != null) {
            this.drawableFaces.get(slot).show(show);
        }
    }

    /**
     * Showing or hiding the feature points
     *
     * @param slot Slot of the face
     * @param show determines whether to hide or show the object
     */
    public void showFeaturePoints(int slot, boolean show) {
        if (slot >= 0 && slot < getArraySize() && this.drawableFeaturePoints.get(slot) != null) {
            this.drawableFeaturePoints.get(slot).show(show);
        }
    }
    
    /**
     * Showing or hiding the symmetry plane
     *
     * @param slot Slot of the face
     * @param show determines whether to hide or show the object
     */
    public void showSymmetryPlane(int slot, boolean show) {
        if (slot >= 0 && slot < getArraySize() && this.drawableSymmetryPlanes.get(slot) != null) {
            this.drawableSymmetryPlanes.get(slot).show(show);
        }
    }

    /**
     * Returns other drawable object.
     * 
     * @param slot Slot of the drawable object
     * @return drawable object or {@code null}
     */
    public Drawable getOtherDrawable(int slot) {
        return (slot >= 0 && slot < this.otherDrawables.size()) ? this.otherDrawables.get(slot) : null;
    }
    
    /**
     * Returns all drawable objects.
     * 
     * @return all drawable objects.
     */
    public List<Drawable> getAllDrawables() {
        List<Drawable> ret = new ArrayList<>();
        ret.addAll(this.drawableFaces);
        ret.addAll(this.drawableFeaturePoints);
        ret.addAll(this.drawableSymmetryPlanes);
        ret.addAll(this.otherDrawables);
        while (ret.remove(null)) {}
        return ret;
    }
    
    public List<DrawableFace> getDrawableFaces() {
        return Collections.unmodifiableList(this.drawableFaces);
    }
    
    protected Color getColorOfFeaturePoints(Color origColor) {
        return new Color(
                Math.min(255, origColor.getRed() + 40),
                Math.min(255, origColor.getGreen() + 40),
                Math.min(255, origColor.getBlue() + 40)
        );
    }
    
    protected boolean prepareArrays(int slot, Object drawable) {
        if (slot >= Scene.MAX_FACES_IN_SCENE) {
            return false;
        }
        
        // extend the 
        for (int i = getArraySize(); i <= slot; i++) {
            this.drawableFaces.add(null);
            this.drawableFeaturePoints.add(null);
            this.drawableSymmetryPlanes.add(null);
            //this.drawableCuttingPlanes.add(null);
        }
        
        return true;
    }

    protected int getArraySize() {
        return this.drawableFaces.size();
    }
    
    @Override
    public String toString() {
        return this.drawableFaces.toString();
    }
}
