package cz.fidentis.analyst.curvature;

import cz.fidentis.analyst.canvas.Canvas;
import cz.fidentis.analyst.core.ControlPanelAction;
import cz.fidentis.analyst.mesh.core.MeshFacet;
import java.awt.event.ActionEvent;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import javax.swing.JComboBox;
import javax.swing.JTabbedPane;
import javax.swing.JToggleButton;

/**
 * Action listener for the curvature computation.
 * 
 * @author Radek Oslejsek
 */
public class CurvatureAction extends ControlPanelAction {
    
    /*
     * Attributes handling the state
     */
    private String curvatureType = CurvaturePanel.GAUSSIAN_CURVATURE;

    private final CurvaturePanel controlPanel;
    private final JTabbedPane topControlPanel;
    
    /**
     * Constructor.
     * 
     * @param canvas OpenGL canvas
     * @param topControlPanel Top component for placing control panels
     */
    public CurvatureAction(Canvas canvas, JTabbedPane topControlPanel) {
        super(canvas, topControlPanel);
        this.topControlPanel = topControlPanel;
        this.controlPanel = new CurvaturePanel(this);

        // Place control panel to the topControlPanel
        this.topControlPanel.addTab(controlPanel.getName(), controlPanel.getIcon(), controlPanel);
        this.topControlPanel.addChangeListener(e -> {
            // If the symmetry panel is focused...
            if (((JTabbedPane) e.getSource()).getSelectedComponent() instanceof CurvaturePanel) {
                getCanvas().getScene().setDefaultColors();
            }
        });

        topControlPanel.setSelectedComponent(controlPanel);
    }

    @Override
    public void actionPerformed(ActionEvent ae) {
        String action = ae.getActionCommand();
        
        switch (action) {
            case CurvaturePanel.ACTION_COMMAND_SHOW_HIDE_PANEL:
                hideShowPanelActionPerformed(ae, this.controlPanel);
                break;
            case CurvaturePanel.ACTION_COMMAND_SHOW_HIDE_HEATMAP:
                if (((JToggleButton) ae.getSource()).isSelected()) {
                    setHeatmap();
                    getPrimaryDrawableFace().setRenderHeatmap(true);
                } else {
                    getPrimaryDrawableFace().setRenderHeatmap(false);
                }
                break;
            case CurvaturePanel.ACTION_COMMAND_SET_CURVATURE_TYPE:
                this.curvatureType = (String) ((JComboBox) ae.getSource()).getSelectedItem();
                setHeatmap();
                break;
            default:
                // to nothing
        }
        
        renderScene();
    }
    
    protected void setHeatmap() {
        getPrimaryDrawableFace().getHumanFace().computeCurvature(false);
        Map<MeshFacet, List<Double>> heatmap = new HashMap<>();
        for (var facet: getPrimaryDrawableFace().getModel().getFacets()) {
            List<Double> vals = facet.getVertices().stream()
                    .map(mp -> {
                        switch (this.curvatureType) {
                            case CurvaturePanel.GAUSSIAN_CURVATURE:
                                return mp.getCurvature().getGaussian();
                            case CurvaturePanel.MEAN_CURVATURE:
                                return mp.getCurvature().getMean();
                            case CurvaturePanel.MIN_CURVATURE:
                                return mp.getCurvature().getMinPrincipal();
                            case CurvaturePanel.MAX_CURVATURE:
                                return mp.getCurvature().getMaxPrincipal();
                            default:
                                throw new UnsupportedOperationException(curvatureType);
                        }
                    })
                    .collect(Collectors.toList());
            heatmap.put(facet, vals);
        }
        getPrimaryDrawableFace().setHeatMap(heatmap);
    }    
}
