package cz.fidentis.analyst.visitors.mesh.sampling;

import cz.fidentis.analyst.grid.UniformGrid3d;
import cz.fidentis.analyst.mesh.core.MeshFacet;
import cz.fidentis.analyst.mesh.core.MeshPoint;
import cz.fidentis.analyst.mesh.core.MeshPointImpl;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;
import javax.vecmath.Point3d;

/**
 * This downsampling algorithm divides the space uniformly using 3D grid.
 * Then, it computes and returns average position (centroid) of points in grid cells.
 * The number of samples is often slightly higher the the required number.
 * 
 * @author Radek Oslejsek
 */
public class UniformSpaceSampling extends PointSampling {
    
    
    private static final double CELL_RESIZE_INIT_VALUE = 1.0;
    private static final double CELL_RESIZE_INIT_STEP = 1.0;
    private static final double CELL_RESIZE_STEP_CHANGE = 2.0;
    private static final double DOWNSAMPLING_TOLERANCE = 50;
    private static final double MAX_ITERATIONS = 100;
    
    private final List<MeshPoint> allVertices = new ArrayList<>();
    
    /** 
     * Constructor for no point sampling (100 %).
     */
    public UniformSpaceSampling() {
        this(1.0);
    }
    
    /**
     * Constructor for PERCENTAGE point sampling type.
     * 
     * @param perc Percentage - a value in (0.0, 1.0&gt;
     * @throws IllegalArgumentException if the input parameter is wrong
     */
    public UniformSpaceSampling(double perc) {
        super(perc);
    }
    
    /**
     * Constructor for MAX_VERTICES point sampling type.
     * 
     * @param max Required number of samples. Must be bigger than zero
     * @throws IllegalArgumentException if the input parameter is wrong
     */
    public UniformSpaceSampling(int max) {
        super(max);
    }
    
    @Override
    public boolean isBackedByOrigMesh() {
        return false;
    }
    
    @Override
    public boolean usesSpaceOrdering() {
        return true;
    }
     
    @Override
    public void visitMeshFacet(MeshFacet facet) {
        if (facet != null) {
            allVertices.addAll(facet.getVertices());
        }
    }
    
    @Override
    public List<MeshPoint> getSamples() {
        // no downsampling:
        if (allVertices.size() == getNumDownsampledPoints(allVertices.size())) { 
            // this sampling method is NOT backed by the original mesh. Therefore, meke a copy.
            return allVertices.stream()
                    .map(p -> new MeshPointImpl(p))                    
                    .collect(Collectors.toList());
        }
        
        // compute controid of cell's points:
        return createGrid().getNonEmptyCells().stream()
                .map(list -> new MeshPointImpl(list)) 
                .collect(Collectors.toList());
    }
    
    @Override
    public String toString() {
        return "uniform space " + super.toString();
    }
    
    protected List<MeshPoint> getOrigPoints() {
        return this.allVertices;
    }
    
    /**
     * Computes the average distance between vertices and their centroid.
     * 
     * @param vertices vertices
     * @param centroid their centroid
     * @return the average distance between vertices and their centroid.
     */
    protected double getAvgDist(Collection<MeshPoint> vertices, Point3d centroid) {
        return vertices.stream()
                .map(p -> p.getPosition())
                .mapToDouble(p -> centroid.distance(p))
                .average()
                .orElse(0);
    }
    
    protected UniformGrid3d<MeshPoint> createGrid() {
        int numReducedVertices = getNumDownsampledPoints(allVertices.size());
        double k = CELL_RESIZE_INIT_VALUE;
        double step = CELL_RESIZE_INIT_STEP;
        int numCells = 0;
        UniformGrid3d<MeshPoint> grid;
        
        MeshPoint centroid = new MeshPointImpl(allVertices);
        double avgDist = getAvgDist(allVertices, centroid.getPosition());
        
        int counter = 0;
        do {
            double cellSize = avgDist / k;
            grid =  new UniformGrid3d<>(cellSize, allVertices, (MeshPoint mp) -> mp.getPosition());
            numCells = grid.numOccupiedCells();
            if (step > 0 && numCells > numReducedVertices) {
                step /= -CELL_RESIZE_STEP_CHANGE;
            } else if (step < 0 && numCells < numReducedVertices) {
                step /= -CELL_RESIZE_STEP_CHANGE;
            } else if (step == 0) {
                break;
            }
            if (counter++ > MAX_ITERATIONS) {
                break;
            }
            k += step;
        } while (numCells < numReducedVertices || numCells > numReducedVertices + DOWNSAMPLING_TOLERANCE);
        
        return grid;
    }
}