package cz.fidentis.analyst.visitors.mesh.sampling;

import cz.fidentis.analyst.mesh.MeshVisitor;
import cz.fidentis.analyst.mesh.core.MeshFacet;
import cz.fidentis.analyst.mesh.core.MeshPoint;
import java.util.List;

/**
 * Point sampling strategies used to reduce the size of meshes.
 * 
 * @author Radek Oslejsek
 */
public abstract class PointSampling extends MeshVisitor {

    /**
     * If point sampling is used, then this configuration parameter 
     * encodes whether the reduction of mesh size is expressed as the percentage 
     * of the original size (number of vertices) or as the maximal number of vertices
     * to be used.
     * 
     * @author Radek Oslejsek
     */
    public enum PointSamplingType {
        PERCENTAGE,
        MAX_VERTICES
    };
    
    private PointSamplingType samplingType;
    private double samplingLimit;
    
    /** 
     * Constructor for no point sampling.
     */
    public PointSampling() {
        this.samplingType = PointSamplingType.PERCENTAGE;
        this.samplingLimit = 1.0;
    }
    
    /**
     * Constructor for PERCENTAGE point sampling type.
     * 
     * @param perc Percentage - a value in (0.0, 1.0&gt;
     * @throws IllegalArgumentException if the input parameter is wrong
     */
    public PointSampling(double perc) {
        setRequiredSamples(perc);
    }
    
    /**
     * Constructor for MAX_VERTICES point sampling type.
     * 
     * @param max Maximal number of vertices. Must be bigger than zero
     * @throws IllegalArgumentException if the input parameter is wrong
     */
    public PointSampling(int max) {
        setRequiredSamples(max);
    }
    
    @Override
    public abstract void visitMeshFacet(MeshFacet facet);    
    
    /**
     * Returns a list of vertices reduced according to the strategy. 
     * The returned mesh points are backed by original points.
     * 
     * @return selected vertices of inspected meshes
     */
    public abstract List<MeshPoint> getSamples();
    
    /**
     * If {@code true}, then the returned points samples are points
     * from the original mesh. Therefore, any change changes the original mesh!
     * If {@code false}, then new points are returned that are independent on the original mesh.
     * 
     * @return {@code true} if the point samples include points of the original mesh
     */
    public boolean isBackedByOrigMesh() {
        return true;
    }
    
    /**
     * If {@code true}, then the returned points samples are stored a space-ordering
     * structure. Therefore, their 3D transformation requires to re-arrange the
     * ordering structure (i.e., to reg-generate samples by calling {@link #getSamples()}
     * again.
     * 
     * from the original mesh. Therefore, any change changes the original mesh!
     * If {@code}, then new points are returned.
     * 
     * @return {@code true} if the point samples include points of the original mesh
     */
    public boolean usesSpaceOrdering() {
        return false;
    }
    
    /**
     * Changes the number of required samples.
     * 
     * @param max Maximal number of vertices. Must be bigger than zero
     */
    public final void setRequiredSamples(int max) {
        if (max <= 0) {
            throw new IllegalArgumentException("max");
        }
        this.samplingType = PointSamplingType.MAX_VERTICES;
        this.samplingLimit = max;
    }
    
    /**
     * Changes the number of required samples.
     * 
     * @param perc Percentage - a value in (0.0, 1.0&gt;
     */
    public final void setRequiredSamples(double perc) {
        if (perc <= 0.0 || perc > 1) {
            throw new IllegalArgumentException("perc = " + perc);
        }
        this.samplingType = PointSamplingType.PERCENTAGE;
        this.samplingLimit = perc;
    }
    
    @Override
    public String toString() {
        if (this.samplingType == PointSamplingType.PERCENTAGE) {
            return "downsampling to " + (int) (samplingLimit * 100) + "%";
        } else {
            return "downsampling to " + (int) samplingLimit + " points";
        }
    }
    
    /**
     * Returns number of points to be returned after downsampling.
     * 
     * @param origPoints Original number of vertices
     * @return number of points to be returned after downsampling.
     */
    protected int getNumDownsampledPoints(int origPoints) {
        switch (this.samplingType) {
            case PERCENTAGE:
                return (int) (origPoints * this.samplingLimit);
            case MAX_VERTICES:
                //nt limit = (int) this.undersamplingLimit;
                //return (limit <= origVertices) ? limit : origVertices;
                return Math.min((int) this.samplingLimit, origPoints);
            default:
                return 0;
        }
    }
    
    protected PointSamplingType getSamplingType() {
        return this.samplingType;
    }
    
    protected double getSamplingLimit() {
        return this.samplingLimit;
    }
}
