package cz.fidentis.analyst.core;

import cz.fidentis.analyst.canvas.Canvas;
import cz.fidentis.analyst.scene.DrawableFace;
import cz.fidentis.analyst.scene.DrawableFeaturePoints;
import cz.fidentis.analyst.scene.Scene;
import java.awt.event.ActionEvent;
import javax.swing.AbstractAction;
import javax.swing.JTabbedPane;
import javax.swing.JToggleButton;

/**
 * Default action listener used to connect specific control panel with the rest of
 * the analytical tab, i.e. canvas and toolbar(s)
 * 
 * @author Radek Oslejsek
 */
public abstract class ControlPanelAction extends AbstractAction {
    
    /*
     * Top component - the space delimited for placing control panels
     */
    private final JTabbedPane topControlPanel;
    
    /**
     * Canvas component
     */
    private final Canvas canvas;
    
    /**
     * Constructor. 
     * 
     * @param canvas OpenGL canvas
     * @param topControlPanel Top component for placing control panels
     * @throws IllegalArgumentException if some param is missing
     */
    public ControlPanelAction(Canvas canvas, JTabbedPane topControlPanel) {
        if (canvas == null) {
            throw new IllegalArgumentException("canvas");
        }
        if (topControlPanel == null) {
            throw new IllegalArgumentException("topControlPanel");
        }
        this.canvas = canvas;
        this.topControlPanel = topControlPanel;
    }
    
    @Override
    public abstract void actionPerformed(ActionEvent ae);
    
    public Canvas getCanvas() {
        return canvas;
    }
    
    /**
     * The generic code for showing/hiding the control panel
     * 
     * @param ae Action event
     */
    protected void hideShowPanelActionPerformed(ActionEvent ae, ControlPanel controlPanel) {
        if (((JToggleButton) ae.getSource()).isSelected()) {
            topControlPanel.addTab(controlPanel.getName(), controlPanel.getIcon(), controlPanel);
            topControlPanel.setSelectedComponent(controlPanel); // focus
        } else {
            topControlPanel.remove(controlPanel);
        }
    }
    
    protected Scene getScene() {
        return canvas.getScene();
    }
    
    protected DrawableFace getPrimaryDrawableFace() {
        return (canvas.getScene() != null) ? canvas.getScene().getDrawableFace(0) : null;
    }
    
    protected DrawableFace getSecondaryDrawableFace() {
        return (canvas.getScene() != null) ? canvas.getScene().getDrawableFace(1) : null;
    }
    
    protected DrawableFeaturePoints getPrimaryFeaturePoints() {
        return (canvas.getScene() != null) ? canvas.getScene().getDrawableFeaturePoints(0) : null;
    }
    
    protected DrawableFeaturePoints getSecondaryFeaturePoints() {
        return (canvas.getScene() != null) ? canvas.getScene().getDrawableFeaturePoints(1) : null;
    }

    protected void renderScene() {
        canvas.renderScene();
    }
    
}
