package cz.fidentis.analyst.distance;

import cz.fidentis.analyst.core.ComboSliderDouble;
import cz.fidentis.analyst.core.LoadedActionEvent;
import cz.fidentis.analyst.feature.FeaturePoint;
import cz.fidentis.analyst.feature.FeaturePointType;
import cz.fidentis.analyst.scene.DrawableFpWeights;
import java.awt.Font;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.swing.AbstractAction;
import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JPanel;

/**
 * @author Radek Oslejsek
 * @author Daniel Schramm
 */
public class FeaturePointsPanel extends JPanel {
    
    public static final String ACTION_COMMAND_FEATURE_POINT_HOVER_IN = "highlight hovered FP";
    public static final String ACTION_COMMAND_FEATURE_POINT_HOVER_OUT = "set default color of hovered FP";
    public static final String ACTION_COMMAND_FEATURE_POINT_SELECT = "highlight feature point with color";
    public static final String ACTION_COMMAND_FEATURE_POINT_RESIZE = "set size of feature point";
    public static final String ACTION_COMMAND_DISTANCE_RECOMPUTE = "recompute the Hausdorff distance";
    
    private Map<FeaturePointType, JLabel> featurePointStats;
    private List<JCheckBox> featurePointCheckBoxes;
    private List<ComboSliderDouble> comboSliders;
    
    /**
     * Constructor.
     * 
     * @param action Action listener
     * @param featurePoints List of all feature points which can be used to calculate
     *                      the weighted Hausdorff distance
     * @param selectedFPs Set of feature point types which are initially selected
     *                    to be used to calculate the weighted Hausdorff distance
     */
    public void initComponents(ActionListener action, List<FeaturePoint> featurePoints, Set<FeaturePointType> selectedFPs) {
        setLayout(new GridBagLayout());
        
        //final ControlPanelBuilder fpBuilder = new ControlPanelBuilder(this);
        
        featurePointStats = new HashMap<>(featurePoints.size());
        featurePointCheckBoxes = new ArrayList<>(featurePoints.size());
        comboSliders = new ArrayList<>(featurePoints.size());
        
        for (int i = 0; i < featurePoints.size(); i++) {
            final FeaturePoint featurePoint = featurePoints.get(i);
            final FeaturePointType featurePointType = featurePoint.getFeaturePointType();
            addRow(i, selectedFPs.contains(featurePointType), action, featurePointType);
        }
    }
    
    /**
     * Updates GUI elements that display the weights of feature points
     * used to calculate the weighted Hausdorff distance.
     * 
     * @param featurePointWeights Map of feature point types and their weights
     */
    public void updateFeaturePointWeights(Map<FeaturePointType, Double> featurePointWeights) {
        featurePointStats.forEach((fpType, fpWeightLabel) -> {
            final Double fpWeight = featurePointWeights.get(fpType);
            fpWeightLabel.setText(fpWeight == null ? "" : String.format("%.3f", fpWeight));
            fpWeightLabel.setFont(new Font("Arial", 1, 12));
        });
    }
    
    /**
     * (De)selects all feature points for the computation of the weighted Hausdorff distance.
     * 
     * @param selected {@code true} if all feature point checkboxes are to be selected,
     *                 {@code false} otherwise
     */
    public void selectAllFeaturePoints(boolean selected) {
        featurePointCheckBoxes.forEach(fpCheckBox ->
            fpCheckBox.setSelected(selected)
        );
    }
    
    /**
     * (De)selects given feature point.
     * 
     * @param index Index of the feature point
     * @param selected {@code true} if all feature point checkboxes are to be selected,
     *                 {@code false} otherwise
     */
    public void selectFeaturePoint(int index, boolean selected) {
        if (index >= 0 && index < featurePointCheckBoxes.size()) {
            featurePointCheckBoxes.get(index).setSelected(selected);
        }
    }
    
    /**
     * Returns current size of the sphere of index-th feature point
     * @param index Index of the feature point
     * @return Sphere size, {@code NaN} on error.
     */
    public double getSphereSize(int index) {
        if (index < 0 || index >= comboSliders.size()) {
            return Double.NaN;
        }
        ComboSliderDouble sl = comboSliders.get(index);
        return sl.getSlider().getValue() / sl.getRecomputationFactor();
    }
    
    /**
     * Creates and returns action listener that can be connected with a low-level 
     * GUI element (e.g., a button). Action event of the low-level element is then
     * re-directed to the given {@code ControlPanelAction} as given command.
     * The listener may also carry additional data as a payload.
     * 
     * @param action An instance of the {@link ControlPanelAction}
     * @param command Control panel command
     * @param data Payload data of the action listener
     * @return Action listener
     */
    protected final ActionListener createListener(ActionListener action, String command, Object data) {
        return new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                action.actionPerformed(new LoadedActionEvent(
                        e.getSource(), 
                        ActionEvent.ACTION_PERFORMED, 
                        command,
                        data)
                ); 
            }  
        };
    }
    
    protected void addRow(int row, boolean selected, ActionListener action, FeaturePointType featurePointType) {
        GridBagConstraints c1 = new GridBagConstraints();  
        c1.insets = new Insets(0, 0, 0, 0);
        c1.gridwidth = 1;
        c1.gridx = 0;
        c1.gridy = row;
        c1.anchor = GridBagConstraints.CENTER;
        c1.fill = GridBagConstraints.NONE;

        JCheckBox checkBox = new JCheckBox();
        checkBox.setSelected(selected);
        checkBox.setText(featurePointType.getName());
        add(checkBox, c1);
        featurePointCheckBoxes.add(checkBox);
        
        checkBox.addActionListener(createListener(action, ACTION_COMMAND_FEATURE_POINT_SELECT, row));
        
        final int index = row;
        checkBox.addMouseListener(new MouseAdapter() { // highlight spheres on mouse hover
            @Override
            public void mouseEntered(MouseEvent e) {
                action.actionPerformed(new LoadedActionEvent(
                        e.getSource(),
                        ActionEvent.ACTION_PERFORMED,
                        ACTION_COMMAND_FEATURE_POINT_HOVER_IN,
                        index));
            }

            @Override
            public void mouseExited(MouseEvent e) {
                action.actionPerformed(new LoadedActionEvent(
                        e.getSource(),
                        ActionEvent.ACTION_PERFORMED,
                        ACTION_COMMAND_FEATURE_POINT_HOVER_OUT,
                        index));
            }
        });
        
        
        
        GridBagConstraints c2 = new GridBagConstraints();  
        //c2.insets = new Insets(0, 25, 0, 0);
        c2.gridwidth = 1;
        c2.gridx = 1;
        c2.gridy = row;
        c2.anchor = GridBagConstraints.CENTER;
        c2.fill = GridBagConstraints.NONE;
        c2.insets = new Insets(0, 0, 0, 0);

        final ComboSliderDouble slider = new ComboSliderDouble(); // 0.0 .. 1.0 by default
        slider.setContinousSync(false);
        slider.setRange(0, 100, 2);
        slider.setValue(DrawableFpWeights.FPW_DEFAULT_SIZE);
        
        //slider.addInputFieldListener(createListener(action, ACTION_COMMAND_FEATURE_POINT_RESIZE, row));
        slider.addSliderListener(createListener(action, ACTION_COMMAND_FEATURE_POINT_RESIZE, row));
        
        slider.addInputFieldListener(
                (ActionEvent e) -> {
                    if (!checkBox.isSelected()) {
                        return; // Recompute only if the feature point is selected
                    }
                    action.actionPerformed(new ActionEvent(
                            e.getSource(),
                            ActionEvent.ACTION_PERFORMED,
                            ACTION_COMMAND_DISTANCE_RECOMPUTE
                    ));
                }
        );
        add(slider, c2);
        comboSliders.add(slider);
        
        
        GridBagConstraints c3 = new GridBagConstraints();
        c3.gridwidth = GridBagConstraints.REMAINDER;
        c3.gridx = 2;
        c3.gridy = row;
        c3.anchor = GridBagConstraints.LINE_START;
        c3.fill = GridBagConstraints.NONE;
        c3.insets = new Insets(0, 10, 0, 0);
        
        JLabel label = new JLabel();
        add(label, c3);
        featurePointStats.put(featurePointType, label);
    }
}
