package cz.fidentis.analyst.core;

import java.math.RoundingMode;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.Locale;
import javax.swing.text.DefaultFormatterFactory;
import javax.swing.text.NumberFormatter;

/**
 * ComboSlider with real numbers.
 * The input field handles real number in given precision (number of fraction digits).
 * The slider shows real numbers on an automatic integer scale.
 * 
 * @author Radek Oslejsek
 */
public class ComboSliderDouble extends ComboSlider {
    
    private double min = 0.0;
    private double max = 1.0;
    private int fractionDigits = 2;
    
    /**
     * Constructor. The default scale and precision is from 0.00 to 1.00.
     */
    public ComboSliderDouble() {
        setRange(0.0, 1.0, 2);
    }
    
    /**
     * Sets the slider and input field range.
     * 
     * @param min Minimum value
     * @param max Maximum value. If less than {@code min}, then {@code min} is used instead.
     * @param fractionDigits precision, i.e., the number of allowed digits in the fraction part of real numbers
     */
    public final void setRange(double min, double max, int fractionDigits) {
        this.min = min;
        this.max = (max < min) ? min : max;
        this.fractionDigits = fractionDigits;
        
        NumberFormat format = DecimalFormat.getInstance(Locale.getDefault());
        format.setMinimumFractionDigits(1);
        format.setMaximumFractionDigits(fractionDigits);
        format.setRoundingMode(RoundingMode.HALF_UP);
        
        NumberFormatter formatter = new NumberFormatter(format);
        formatter.setValueClass(Double.class);
        formatter.setMinimum(this.min);
        formatter.setMaximum(this.max);
        formatter.setAllowsInvalid(false);
        
        getInputField().setFormatterFactory(new DefaultFormatterFactory(formatter));
        getInputField().setText(TextFieldUtils.doubleToStringLocale(this.min, fractionDigits));
        
        getSlider().setMinimum((int) (this.min * getRecomputationFactor()));
        getSlider().setMaximum((int) (this.max * getRecomputationFactor()));
        
        setValue(this.min);
    }

    /**
     * Returns the value.
     * 
     * @return the value.
     */
    public double getValue() {
        return TextFieldUtils.parseLocaleDouble(getInputField());
    }

    /**
     * Sets the value of the slider and input field. 
     * If the value is outside of the range then nothing happens.
     * 
     * @param value a new value
     */
    public void setValue(double value) {
        if (value >= getMinimum() && value <= getMaximum()) {
            getInputField().setText(TextFieldUtils.doubleToStringLocale(value, fractionDigits));
            getInputField().postActionEvent(); // invoke textField action listener
        }
    }
    
    /**
     * Returns the lower limit of the range.
     * 
     * @return the lower limit
    */
    public double getMinimum() {
        return this.min;
    }

    /**
     * Returns the upper limit of the range.
     * 
     * @return the upper limit
     */
    public double getMaximum() {
        return this.max;
    }
    
    /**
     * Returns the number of allowed digits in the fraction part of real numbers.
     * 
     * @return the number of allowed digits in the fraction part of real numbers
     */
    public int getFractionDigits() {
        return this.fractionDigits;
    }
    
    @Override
    protected void setValueFromSlider() {
        setValue(getSlider().getValue() / getRecomputationFactor()); // also triggers the event
    }
    
    @Override
    protected void setValueFromInputField() {
        getSlider().setValue((int) (TextFieldUtils.parseLocaleDouble(getInputField()) * getRecomputationFactor()));
    }
    
    /**
     * Returns 1, 10, 100, etc., based on the fraction digits.
     * @return 1, 10, 100, etc., based on the fraction digits.
     */
    public double getRecomputationFactor() {
        int scale = 1;
        for (int i = 0; i < fractionDigits; i++) {
            scale *= 10;
        }
        return scale;
    }
}
