package cz.fidentis.analyst.scene;

import com.jogamp.opengl.GL2;
import com.jogamp.opengl.glu.GLU;
import com.jogamp.opengl.glu.GLUquadric;
import cz.fidentis.analyst.feature.FeaturePoint;
import java.awt.Color;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Drawable feature points.
 * 
 * @author Radek Oslejsek
 */
public class DrawableFeaturePoints extends Drawable {
    
    public static final Color DEFAULT_COLOR = Color.GREEN;
    
    private static final GLU GLU_CONTEXT = new GLU();
    
    /* feature points */
    private final List<FeaturePoint> featurePoints;
    
    /**
     * feature points with color different from the default color
     */
    private Map<Integer, Color> specialColors = new HashMap<>();
    
    /**
     * Constructor.
     * @param featurePoints Feature points
     */
    public DrawableFeaturePoints(List<FeaturePoint> featurePoints) {
        this.featurePoints = new ArrayList<>(featurePoints);
        setColor(DEFAULT_COLOR);
    }
    
    /**
     * Returns color of given feature point
     * @param index Index of the feature point
     * @return the color or {@code null}
     */
    public Color getColor(int index) {
        if (index < 0 || index >= featurePoints.size()) {
            return null;
        }
        if (specialColors.containsKey(index)) {
            return specialColors.get(index);
        } else {
            return DEFAULT_COLOR;
        }
    }
    
    /**
     * Sets color of the feature point. 
     * @param index Index of the feature point
     * @param color Color
     */
    public void setColor(int index, Color color) {
        if (index < 0 || index >= featurePoints.size() || color == null) {
            return;
        }
        if (color.equals(DEFAULT_COLOR)) {
            specialColors.remove(index);
        } else {
            specialColors.put(index, color);
        }
    }

    @Override
    protected void renderObject(GL2 gl) {
        float[] rgba = {
            getColor().getRed() / 255f, 
            getColor().getGreen() / 255f, 
            getColor().getBlue() / 255f, 
            getTransparency()
        };
        gl.glMaterialfv(GL2.GL_FRONT_AND_BACK, GL2.GL_DIFFUSE, rgba, 0); // set default color
        
        for (int i = 0; i < featurePoints.size(); i++) {
            FeaturePoint fp = featurePoints.get(i);
            Color specialColor = specialColors.get(i);
            if (specialColor != null) {
                float[] tmpRGB = {specialColor.getRed() / 255f, specialColor.getGreen() / 255f, specialColor.getBlue() / 255f, getTransparency()};
                gl.glMaterialfv(GL2.GL_FRONT_AND_BACK, GL2.GL_DIFFUSE, tmpRGB, 0);
            }
            
            gl.glPushMatrix(); 
            gl.glTranslated(fp.getX(), fp.getY(), fp.getZ());
            GLUquadric center = GLU_CONTEXT.gluNewQuadric();
            GLU_CONTEXT.gluQuadricDrawStyle(center, GLU.GLU_FILL);
            GLU_CONTEXT.gluQuadricNormals(center, GLU.GLU_FLAT);
            GLU_CONTEXT.gluQuadricOrientation(center, GLU.GLU_OUTSIDE);
            GLU_CONTEXT.gluSphere(center, 3f, 16, 16);
            GLU_CONTEXT.gluDeleteQuadric(center);
            gl.glPopMatrix();
    
            if (specialColor != null) {
                gl.glMaterialfv(GL2.GL_FRONT_AND_BACK, GL2.GL_DIFFUSE, rgba, 0); // set default color
            }
        }
    }
    
    
       
    /**
     * @return {@link List} of {@link FeaturePoint}
     */
    public List<FeaturePoint> getFeaturePoints() {
        return Collections.unmodifiableList(featurePoints);
    }

    /**
     * Sets feature points
     * @param featurePoints Feature points
     */
    /*
    public void setFeaturePoints(List<FeaturePoint> featurePoints) {
        this.featurePoints = featurePoints;
        List<Color> colors = new ArrayList<>();
        featurePoints.forEach((_item) -> {
            colors.add(getColor());
        });
        this.setFeaturePointsColor(colors);
    } 
    */
    
}
