package cz.fidentis.analyst.canvas;

import com.jogamp.opengl.GLCapabilities;
import com.jogamp.opengl.GLProfile;
import com.jogamp.opengl.awt.GLCanvas;
import cz.fidentis.analyst.face.HumanFace;
import cz.fidentis.analyst.scene.Camera;
import cz.fidentis.analyst.scene.Scene;
import cz.fidentis.analyst.scene.SceneRenderer;
import java.awt.BorderLayout;

import javax.swing.JLayeredPane;
import javax.swing.JPanel;

/**
 * OpenGL canvas.
 * 
 * @author Natalia Bebjakova 
 * @author Radek Oslejsek
 */
public class Canvas extends JPanel {
    
    private GLCanvas glCanvas;
    private final SceneRenderer sceneRenderer;
    private final Camera camera = new Camera();
    private Scene scene;
    
    // Listeners:
    private final CanvasListener listener;
    private final MouseRotationListener manipulator;
    private final RotationAnimator animator;
    
    private ControlButtons controlButtonsPanel;

    private JLayeredPane layeredPane;
    private JPanel canvasPanel;

    
    /**
     * Constructor.
     */
    public Canvas() {
        sceneRenderer = new SceneRenderer();
        listener = new CanvasListener(this);
        manipulator = new MouseRotationListener(this);
        
        setLayout(new BorderLayout());

        initGLCanvas();
        canvasPanel = getCanvasPanel();
        canvasPanel.add(glCanvas);
        
        controlButtonsPanel = getButtonsPanel();
        
        layeredPane = getLayeredPane();
        layeredPane.add(canvasPanel, 1);
        //layeredPane.setLayer(controlButtonsPanel, JLayeredPane.MODAL_LAYER);
        layeredPane.add(controlButtonsPanel, 0);
        add(layeredPane, BorderLayout.CENTER);
        
        setDarkBackground(true);
        
        animator = new RotationAnimator(this.glCanvas);

        validate();   
    }
    
    private ControlButtons getButtonsPanel() {
        ControlButtonsAction controlButtonsListener = new ControlButtonsAction();
        ControlButtons ret = new ControlButtons(controlButtonsListener);
        ret.setOpaque(false);
        ret.setBounds(20, 20, ControlButtons.SIZE.width, ControlButtons.SIZE.height);
        return ret;
    }
    
    private JLayeredPane getLayeredPane() {
        JLayeredPane pane = new JLayeredPane();
        pane.addComponentListener(new java.awt.event.ComponentAdapter() {
            @Override
            public void componentResized(java.awt.event.ComponentEvent evt) {
                jLayeredPane1ComponentResized(evt);
            }
            @Override
            public void componentShown(java.awt.event.ComponentEvent evt) {
                jLayeredPane1ComponentShown(evt);
            }
        });
        return pane;
    }
    
    private JPanel getCanvasPanel() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());
        panel.setBounds(0, 0, 0, 0);
        return panel;
    }
    
    private void initGLCanvas() {
        // gl version 2 is used
        GLCapabilities capabilities = new GLCapabilities(GLProfile.get(GLProfile.GL2));
        capabilities.setDoubleBuffered(true);
       
        // creates new glCanvas panel for displaying model
        glCanvas = new GLCanvas(capabilities);
        glCanvas.setVisible(true);     
        
        // enables glCanvas to react to events
        glCanvas.requestFocusInWindow();        
        glCanvas.setSize(getWidth() - getInsets().left - getInsets().right, getHeight() - getInsets().top - getInsets().bottom);

        glCanvas.addGLEventListener(listener);

        glCanvas.addMouseListener(manipulator);
        glCanvas.addMouseMotionListener(manipulator);
        glCanvas.addMouseWheelListener(manipulator);        
    }
    
    /**
     * Initializes the scene for a single face. Can be called only once.
     * 
     * @param face Human face
     * @throws UnsupportedOperationException the the scene is already initiated
     * @throws IllegalArgumentException if the face is missing
     */
    public void initScene(HumanFace face) {
        if (scene != null) {
            throw new UnsupportedOperationException("Scene already exists");
        }
        this.scene = new Scene(face);
    }
    
    /**
     * Initializes the scene for the 1:1 comparison. Can be called only once.
     * 
     * @param primary Primary face
     * @param secondary Secondary face
     * @throws UnsupportedOperationException the the scene is already initiated
     * @throws IllegalArgumentException if some face is missing
     */
    public void initScene(HumanFace primary, HumanFace secondary) {
        if (scene != null) {
            throw new UnsupportedOperationException("Scene already exists");
        }
        this.scene = new Scene(primary, secondary);
    }
   
    /**
     * Returns the scene
     * @return the scene
     */
    public SceneRenderer getSceneRenderer() {
        return sceneRenderer;
    }
    
    /**
     * Renders the scene.
     */
    public void renderScene() {
        glCanvas.display();
    }
    
    /**
     * Returns the scene. 
     * 
     * @return the scene.
     */
    public Scene getScene() {
        return scene;
    }
    
    /**
     * Returns camera. 
     * 
     * @return camera
     */
    public Camera getCamera() {
        return camera;
    }
    
    /**
     * Sets background color
     * 
     * @param dark If {@code true}, then dark background is set
     */
    public void setDarkBackground(boolean dark) {
        if (dark) {
            canvasPanel.setBackground(SceneRenderer.DARK_BACKGROUND);
            sceneRenderer.setDarkBackground();
        } else {
            canvasPanel.setBackground(SceneRenderer.BRIGHT_BACKGROUND);
            sceneRenderer.setBrightBackground();
        }
    }
    
    /**
     * Returns the underlying OpenGL canvas.
     * 
     * @return the underlying OpenGL canvas.
     */
    /*
    public GLCanvas getGLCanvas() {
        return this.glCanvas;
    }*/
    
    
    /**
     * Changing the size of glCanvas
     * 
     * @param d New size of glCanvas
     */
    /*
    public void resizeCanvas(Dimension d) {
        jPanel1.setSize(d);
        glCanvas.setSize(d);
        this.validate();
        this.repaint();
    }
    */
    
    private void jLayeredPane1ComponentResized(java.awt.event.ComponentEvent evt) {//GEN-FIRST:event_jLayeredPane1ComponentResized
        canvasPanel.setBounds(0, 0, layeredPane.getWidth(), layeredPane.getHeight());
        glCanvas.setBounds(layeredPane.getX(), layeredPane.getY(), layeredPane.getWidth(), layeredPane.getHeight());
    }//GEN-LAST:event_jLayeredPane1ComponentResized

    private void jLayeredPane1ComponentShown(java.awt.event.ComponentEvent evt) {//GEN-FIRST:event_jLayeredPane1ComponentShown
        canvasPanel.setBounds(0, 0, layeredPane.getWidth(), layeredPane.getHeight());
        glCanvas.setBounds(layeredPane.getX(), layeredPane.getY(), layeredPane.getWidth(), layeredPane.getHeight());
    }//GEN-LAST:event_jLayeredPane1ComponentShown

    private void leftNavigationButtonMousePressed(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_leftNavigationButtonMousePressed
        animator.startAnimation(Direction.ROTATE_LEFT, this.camera);
        //leftNavigationButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/leftButtonPressed.png")));
    }//GEN-LAST:event_leftNavigationButtonMousePressed

    private void upNavigationButtonMousePressed(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_upNavigationButtonMousePressed
        animator.startAnimation(Direction.ROTATE_UP, this.camera);
        //upNavigationButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/upButtonPressed.png")));
    }//GEN-LAST:event_upNavigationButtonMousePressed

    private void downNavigationButtonMousePressed(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_downNavigationButtonMousePressed
        animator.startAnimation(Direction.ROTATE_DOWN, this.camera);
        //downNavigationButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/downButtonPressed.png")));
    }//GEN-LAST:event_downNavigationButtonMousePressed

    private void plusNavigationButtonMousePressed(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_plusNavigationButtonMousePressed
        animator.startAnimation(Direction.ZOOM_IN, this.camera);
        //plusNavigationButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/plusPressed.png")));
    }//GEN-LAST:event_plusNavigationButtonMousePressed
 
    private void minusNavigationButtonMousePressed(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_minusNavigationButtonMousePressed
        animator.startAnimation(Direction.ZOOM_OUT, this.camera);
        //minusNavigationButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/minusPressed.png")));
    }//GEN-LAST:event_minusNavigationButtonMousePressed

    private void leftNavigationButtonMouseReleased(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_leftNavigationButtonMouseReleased
        animator.stopAnimation(this.camera);
        //leftNavigationButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/leftButton.png")));
    }//GEN-LAST:event_leftNavigationButtonMouseReleased

    private void upNavigationButtonMouseReleased(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_upNavigationButtonMouseReleased
        animator.stopAnimation(this.camera);
        //upNavigationButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/upButton.png")));
    }//GEN-LAST:event_upNavigationButtonMouseReleased

    private void downNavigationButtonMouseReleased(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_downNavigationButtonMouseReleased
        animator.stopAnimation(this.camera);
        //downNavigationButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/downButton.png")));
    }//GEN-LAST:event_downNavigationButtonMouseReleased

    private void plusNavigationButtonMouseReleased(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_plusNavigationButtonMouseReleased
        animator.stopAnimation(this.camera);
        //plusNavigationButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/plus.png")));
    }//GEN-LAST:event_plusNavigationButtonMouseReleased

    private void minusNavigationButtonMouseReleased(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_minusNavigationButtonMouseReleased
        animator.stopAnimation(this.camera);
        //minusNavigationButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/minus.png")));
    }//GEN-LAST:event_minusNavigationButtonMouseReleased

    private void rightNavigationButton1MousePressed(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_rightNavigationButton1MousePressed
        animator.startAnimation(Direction.ROTATE_RIGHT, this.camera);
        //rightNavigationButton1.setIcon(new javax.swing.ImageIcon(getClass().getResource("/rightButtonPressed.png")));
    }//GEN-LAST:event_rightNavigationButton1MousePressed

    private void rightNavigationButton1MouseReleased(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_rightNavigationButton1MouseReleased
        animator.stopAnimation(this.camera);
        //rightNavigationButton1.setIcon(new javax.swing.ImageIcon(getClass().getResource("/rightButton.png")));
    }//GEN-LAST:event_rightNavigationButton1MouseReleased

    private void resetButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_resetButtonMouseClicked
        camera.initLocation();
        renderScene();
    }//GEN-LAST:event_resetButtonMouseClicked

    private void resetButtonMouseMoved(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_resetButtonMouseMoved
        //resetButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/resetButtonPressed.png")));
        //resetButton.setCursor(Cursor.getPredefinedCursor(Cursor.HAND_CURSOR));
    }//GEN-LAST:event_resetButtonMouseMoved

    private void resetButtonMouseExited(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_resetButtonMouseExited
        //resetButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/resetButton.png")));
    }//GEN-LAST:event_resetButtonMouseExited

    // Variables declaration - do not modify//GEN-BEGIN:variables
    //private javax.swing.JButton downNavigationButton;
    //private javax.swing.JLabel jLabel1;
    //private javax.swing.JButton leftNavigationButton;
    //private javax.swing.JButton minusNavigationButton;
    //private javax.swing.JButton plusNavigationButton;
    //private javax.swing.JLabel resetButton;
    //private javax.swing.JButton rightNavigationButton1;
    //private javax.swing.JButton upNavigationButton;
    // End of variables declaration//GEN-END:variables
}
