package cz.fidentis.analyst.mesh.core;

import javax.vecmath.Vector3d;

/**
 * MeshPoint represents a point with position, normal, and texture coordinates.
 *
 * @author Matej Lukes
 */
public class MeshPointImpl implements MeshPoint {
    
    private final Vector3d position, normal, texCoord;

    /**
     * constructor of meshPoint
     *
     * @param position position of MeshPoint
     * @param normal   normal of MeshPoint
     * @param texCoord coordinates in texture
     */
    public MeshPointImpl(Vector3d position, Vector3d normal, Vector3d texCoord) {
        if (position == null) {
            throw new NullPointerException("position cannot be null");
        }

        this.position = new Vector3d(position);
        this.normal = new Vector3d(normal);
        this.texCoord = new Vector3d(texCoord);
    }

    /**
     * copy constructor of meshPoint
     * 
     * @param meshPoint copied meshPoint
     */
    public MeshPointImpl(MeshPoint meshPoint) {
        this(meshPoint.getPosition(), meshPoint.getNormal(), meshPoint.getTexCoord());

    }

    @Override
    public Vector3d getNormal() {
        return normal;
    }

    @Override
    public Vector3d getPosition() {
        return position;
    }

    @Override
    public Vector3d getTexCoord() {
        return texCoord;
    }

    @Override
    public MeshPoint subtractPosition(MeshPoint subtrahend) {
        return subtractPosition(subtrahend.getPosition());
    }

    @Override
    public MeshPoint subtractPosition(Vector3d subtrahend) {
        Vector3d newPosition = new Vector3d(position);
        newPosition.sub(subtrahend);
        return new MeshPointImpl(new Vector3d(newPosition), normal, new Vector3d(texCoord));
    }

    @Override
    public MeshPoint addPosition(MeshPoint addend) {
        return addPosition(addend.getPosition());
    }

    @Override
    public MeshPoint addPosition(Vector3d addend) {
        Vector3d newPosition = new Vector3d(position);
        newPosition.add(addend);
        return new MeshPointImpl(new Vector3d(newPosition), normal, new Vector3d(texCoord));
    }

    @Override
    public MeshPoint multiplyPosition(double number) {
        if (normal != null) {
            if (texCoord != null) {
                return new MeshPointImpl(new Vector3d(this.getPosition().x * number,
                        this.getPosition().y * number, this.getPosition().z * number),
                        new Vector3d(normal), new Vector3d(texCoord));
            } else {
                return new MeshPointImpl(new Vector3d(this.getPosition().x * number,
                        this.getPosition().y * number, this.getPosition().z * number),
                        new Vector3d(normal), null);
            }
        }
        return new MeshPointImpl(new Vector3d(this.getPosition().x * number,
                        this.getPosition().y * number, this.getPosition().z * number),
                        null, null);        
    }

    @Override
    public MeshPoint dividePosition(double number) {
        if (normal != null) {
            if (texCoord != null) {
                return new MeshPointImpl(new Vector3d(this.getPosition().x / number, this.getPosition().y / number,
                        this.getPosition().z / number), new Vector3d(normal), new Vector3d(texCoord));
            } else {
                return new MeshPointImpl(new Vector3d(this.getPosition().x / number, this.getPosition().y / number,
                        this.getPosition().z / number), new Vector3d(normal), null);
            }
        }
        return new MeshPointImpl(new Vector3d(this.getPosition().x / number, this.getPosition().y / number,
                this.getPosition().z / number), null, null);
    }
    
    @Override
    public MeshPoint crossProduct(MeshPoint meshPoint) {
        if (normal != null) {
            if (texCoord != null) {
                return new MeshPointImpl(new Vector3d
                (this.position.y * meshPoint.getPosition().z - this.position.z * meshPoint.getPosition().y,
                this.position.z * meshPoint.getPosition().x - this.position.x * meshPoint.getPosition().z,
                this.position.x * meshPoint.getPosition().y - this.position.y * meshPoint.getPosition().x),
                new Vector3d(normal), new Vector3d(texCoord));
            } else {
                return new MeshPointImpl(new Vector3d
                (this.position.y * meshPoint.getPosition().z - this.position.z * meshPoint.getPosition().y,
                this.position.z * meshPoint.getPosition().x - this.position.x * meshPoint.getPosition().z,
                this.position.x * meshPoint.getPosition().y - this.position.y * meshPoint.getPosition().x),
                new Vector3d(normal), null);
            }
        }
        return new MeshPointImpl(new Vector3d
                (this.position.y * meshPoint.getPosition().z - this.position.z * meshPoint.getPosition().y,
                this.position.z * meshPoint.getPosition().x - this.position.x * meshPoint.getPosition().z,
                this.position.x * meshPoint.getPosition().y - this.position.y * meshPoint.getPosition().x),
                null, null);
    }

    @Override
    public double dotProduct(MeshPoint meshPoint) {
        return (this.position.x * meshPoint.getPosition().x + this.position.y * meshPoint.getPosition().y + this.position.z * meshPoint.getPosition().z);
    }
    
    @Override
    public double abs() {
        return Math.sqrt(this.getPosition().x * this.getPosition().x + 
                this.getPosition().y * this.getPosition().y + this.getPosition().z * this.getPosition().z);
    }

    @Override
    public MeshPoint subtractNormal(MeshPoint subtrahend) {
        return subtractNormal(subtrahend.getNormal());
    }

    @Override
    public MeshPoint subtractNormal(Vector3d subtrahend) {
        Vector3d newNormal = new Vector3d(normal);
        newNormal.sub(subtrahend);
        newNormal.normalize();
        return new MeshPointImpl(new Vector3d(position), newNormal, new Vector3d(texCoord));
    }

    @Override
    public MeshPoint addNormal(MeshPoint addend) {
        return addNormal(addend.getNormal());
    }

    @Override
    public MeshPoint addNormal(Vector3d addend) {
        Vector3d newNormal = new Vector3d(normal);
        newNormal.add(addend);
        newNormal.normalize();
        return new MeshPointImpl(new Vector3d(position), newNormal, new Vector3d(texCoord));
    }

    /**
     * @param obj compared object
     * @return true if positions, normals and texture coordinates are equal, false otherwise
     */
    @Override
    public boolean equals(Object obj) {
        if (!(obj instanceof MeshPointImpl)) {
            return false;
        }

        MeshPointImpl meshPointObj = (MeshPointImpl) obj;
        return this.position.equals(meshPointObj.position)
                && this.normal.equals(meshPointObj.normal)
                && this.texCoord.equals(meshPointObj.texCoord);
    }

    /**
     * returns hash of MeshPoint
     *
     * @return hash
     */
    @Override
    public int hashCode() {
        return position.hashCode() + normal.hashCode() + texCoord.hashCode();
    }

}
