package cz.fidentis.analyst.mesh.core;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * Representation of mesh in memory
 */
/**
 * Representation of mesh in memory using corner table
 *
 * @author Matej Lukes
 */
public class CornerTable {

    private List<CornerTableRow> rows = new ArrayList<>();
    
    /**
     * Constructor of CornerTable
     */
    public CornerTable() {

    }

     /**
     * Copy constructor of CornerTable
     *
     * @param cornerTable copied CornerTable
     */
    public CornerTable(CornerTable cornerTable) {
        cornerTable.rows.forEach((row) -> {
            rows.add(new CornerTableRow(row));
        });
    }


    /**
     * returns index of face that contains corner.
     * returns -2 if index is less than 0 or more than number of rows in corner table
     *
     * @param index index of corner
     * @return index of corner
     */
    public int getIndexOfFace(int index) {
        if (index < 0 || index > rows.size()) {
            return -2;
        }
        return index / 3;
    }

    /**
     * returns index of opposite corner.
     * returns -1 if corner does not have opposite
     * returns -2 if index is less than 0 or more than number of rows in corner table
     *
     * @param index index of corner
     * @return index of corner
     */
    public int getIndexOfOppositeCorner(int index) {
        if (index < 0 || index > rows.size()) {
            return -2;
        }
        return rows.get(index).getOppositeCornerIndex();
    } // opposite

    /**
     * returns index of next corner in Face
     * returns -2 if index is less than 0 or more than number of rows in corner table
     *
     * @param index index of corner
     * @return index of next corner
     */
    public int getIndexOfNextCornerInFace(int index) {
        if (index < 0 || index > rows.size()) {
            return -2;
        }
        if ((index % 3) == 2) {
            return index - 2;
        }
        return index + 1;
    }

    /**
     * returns index of previous corner in Face
     * returns -2 if index is less than 0 or more than number of rows in corner table
     *
     * @param index index of corner
     * @return index of previous corner
     */
    public int getIndexOfPreviousCornerInFace(int index) {
        if (index < 0 || index > rows.size()) {
            return -2;
        }
        return getIndexOfNextCornerInFace(getIndexOfNextCornerInFace(index));
    }

    /**
     * returns opposite corner of edge on the left side of corner
     * returns -2 if index is less than 0 or more than number of rows in corner table
     *
     * @param index index of corner
     * @return index of corner
     */
    public int getIndexOfTipCornerOnLeft(int index) {
        if (index < 0 || index > rows.size()) {
            return -2;
        }
        return getIndexOfOppositeCorner(getIndexOfPreviousCornerInFace(index));
    }

    /**
     * returns opposite corner of edge on the right side of corner
     * returns -2 if index is less than 0 ord more than number of rows in corner table
     *
     * @param index index of corner
     * @return index of corner
     */
    public int getIndexOfTipCornerOnRight(int index) {
        if (index < 0 || index > rows.size()) {
            return -2;
        }
        return getIndexOfOppositeCorner(getIndexOfNextCornerInFace(index));
    }

    /**
     * returns corner next to the corner opposite the edge on the left side
     * returns -2 if index is less than 0 ord more than number of rows in corner table
     *
     * @param index index of corner
     * @return index of corner
     */
    public int getNextAroundCorner(int index) {
        if (index < 0 || index > rows.size()) {
            return -2;
        }
        int tipLeftCornerIndex = getIndexOfTipCornerOnLeft(index);
        if (tipLeftCornerIndex == -1) {
            return -1;
        }

        return getIndexOfNextCornerInFace(tipLeftCornerIndex);
    }

    /**
     * adds row at the end of corner table
     *
     * @param row that is going to be added to corner table
     */
    public void addRow(CornerTableRow row) {
        rows.add(row);
    }

    /**
     * replaces row of corner table at specified index
     *
     * @param index index of replaced row
     * @param row   new row
     */
    public void replaceRow(int index, CornerTableRow row) {
        rows.set(index, row);
    }

    /**
     * returns number of rows in corner table
     *
     * @return number of rows in corner table
     */
    public int getSize() {
        return rows.size();
    }

    /**
     * returns row of corner table with specified index
     *
     * @param index index of desired row
     * @return row of corner table
     */
    public CornerTableRow getRow(int index) {
        return rows.get(index);
    }

    /**
     * returns corners of specific vertex
     *
     * @param vertexIndex index of vertex
     * @return list of rows of corner table
     */
    public List<CornerTableRow> getCornersByVertexIndex(int vertexIndex) {
        return rows.stream()
                .filter(corner -> corner.getVertexIndex() == vertexIndex)
                .collect(Collectors.toList());
    }

    /**
     * returns list of indexes of faces that have a corner at specific vertex
     *
     * @param vertexIndex index of vertex
     * @return list of indexes of faces
     */
    public List<Integer> getTriangleIndexesByVertexIndex(int vertexIndex) {
        return rows.stream()
                .filter(corner -> corner.getVertexIndex() == vertexIndex)
                .map(corner -> getIndexOfFace(rows.indexOf(corner)))
                .collect(Collectors.toList());
    }
    
    /**
     * returns indexes of vertices of triangle
     * @param triangleIndex index of triangle
     * @return list of indexes
     */
    public List<Integer> getIndexesOfVerticesByTriangleIndex(int triangleIndex) {
        List<Integer> indexes = new ArrayList<>();
        for (int i = 0; i < 3; i++) {
            CornerTableRow row = getRow(triangleIndex + i);
            indexes.add(row.getVertexIndex());
        }
        return indexes;
    }

}


