package cz.fidentis.analyst.mesh.io;

import cz.fidentis.analyst.mesh.core.MeshFacet;
import cz.fidentis.analyst.mesh.core.MeshModel;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.Locale;

/**
 * Utility class for exporting model to .obj file (vertices, normals and triangles)
 * 
 * @author Natalia Bebjakova
 */
public class MeshObjExporter {
    /**
     * Model to be exported
     */
    private MeshModel model;
    
    /**
     *
     * @param model model to be exported.
     */
    public MeshObjExporter(MeshModel model) {
        this.model = model;
    }
    
    /**
     * Exports all facets of the model to .obj file
     * 
     * @param exportFile File to which model is exported
     * @throws IOException 
     */
    public void exportModelToObj(File exportFile) throws IOException {
        for(MeshFacet facet : model.getFacets()) {
            exportFacetToObj(facet, exportFile);
        }
    }

    /**
     * Exports facet to OBJ file.
     * It writes vertices, normals and triangles to file
     * 
     * @param facet Facet of the model to be exported, so far every model has one
     * @param exportFile file for exporting.
     * @throws java.io.IOException 
     */
    public void exportFacetToObj(MeshFacet facet, File exportFile) throws IOException {
        int formatIndex = exportFile.getName().lastIndexOf(".");
        String fileName; //name that is writen to file
        
        if (formatIndex < 0) {
            fileName = exportFile.getName();
        } else {
            fileName = exportFile.getName().substring(0, formatIndex);
        }

        exportFile = new File(exportFile.getParent() + File.separator + fileName + ".obj");
        
        try (BufferedWriter out = new BufferedWriter(new FileWriter(exportFile))) {
           
            DecimalFormatSymbols otherSymbols = new DecimalFormatSymbols(Locale.getDefault());
            otherSymbols.setDecimalSeparator('.');  //dot as separator for decimal numbers
            DecimalFormat df = new DecimalFormat("#.####", otherSymbols);
             
            //writes vertices of the facet to file
            for (int j = 0; j < facet.getNumberOfVertices(); j++) {
                out.write("v " + df.format(facet.getVertices().get(j).getPosition().x) + " " 
                        + df.format(facet.getVertices().get(j).getPosition().y) + " " 
                        + df.format(facet.getVertices().get(j).getPosition().z));
                out.newLine();
            }
            
            //detects if first vertex has normal
            boolean hasAllNormals = facet.getVertices().get(0).getNormal() != null;
            
            //writes normals if there are any 
            for (int i = 0; i < facet.getNumberOfVertices(); i++) {
                if(facet.getVertex(i).getNormal() != null) {
                out.write("vn " + df.format(facet.getVertices().get(i).getNormal().x) + " "
                        + df.format(facet.getVertices().get(i).getNormal().y) + " "
                        + df.format(facet.getVertices().get(i).getNormal().z));
                out.newLine();
                }
            }
            
            for (int i = 0; i < facet.getCornerTable().getSize(); i += 3) {
                int v1i = facet.getCornerTable().getRow(i).getVertexIndex();
                int v2i = facet.getCornerTable().getRow(i + 1).getVertexIndex();
                int v3i = facet.getCornerTable().getRow(i + 2).getVertexIndex();
                
                out.write("f ");
                if (facet.getVertex(v1i).getNormal() != null && hasAllNormals) {
                    out.write((v1i + 1) + "//" + (v1i + 1) + " "
                            + (v2i + 1) + "//" + (v2i + 1) + " "
                            + (v3i + 1) + "//" + (v3i + 1)); 
                    out.newLine();
                } else {
                    out.write((v1i + 1) + " " + (v2i + 1) + " " + (v3i + 1));
                    out.newLine();
                }
            }
            
            out.write("#" + facet.getCornerTable().getSize() / 3 + " triangles");
            out.newLine();
            
            //writeTriangles(facet, out);
        }
    }
    
    /*
    protected void writeTriangles(MeshFacet facet, BufferedWriter out) throws IOException {
            Triangle[] triangles = new Triangle[facet.getCornerTable().getSize() / 3];
            for (int i = 0; i < facet.getCornerTable().getSize(); i += 3) {
                Triangle t = new Triangle(facet.getCornerTable().getRow(i).getVertexIndex(),
                    facet.getCornerTable().getRow(i + 1).getVertexIndex(),
                    facet.getCornerTable().getRow(i + 2).getVertexIndex());
                triangles[(i / 3)] = t;
            }
            
            //writes triangles of facet
            for (Triangle triangle : triangles) {
                out.write("f ");
                if (facet.getVertex(triangle.getVertex1()).getNormal() != null && hasAllNormals) {
                    out.write((triangle.getVertex1() + 1) + "//" + (triangle.getVertex1() + 1) + " "
                            + (triangle.getVertex2() + 1) + "//" + (triangle.getVertex2() + 1) + " "
                            + (triangle.getVertex3() + 1) + "//" + (triangle.getVertex3() + 1)); 
                    out.newLine();
                } else {
                    out.write((triangle.getVertex1() + 1) + " " + (triangle.getVertex2() + 1) +
                            " " + (triangle.getVertex3() + 1));   
                    out.newLine();
                }
            }
            out.write("#" + triangles.length + " triangles");
            out.newLine();
            out.newLine();
    }
    */
}

