package cz.fidentis.analyst.mesh.core;

import com.google.common.eventbus.EventBus;
import java.util.ArrayList;
import java.util.List;
import cz.fidentis.analyst.mesh.MeshVisitor;
import cz.fidentis.analyst.mesh.events.FacetAddedEvent;
import java.util.Collections;
import cz.fidentis.analyst.mesh.events.MeshListener;
import java.util.LinkedList;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * The main object for triangular meshes. Each mesh model consists  
 * of one or more mesh facets.
 * <p>
 * This class implements the publish-subscribe notifications to changes. 
 * However, the {@link cz.fidentis.analyst.face.HumanFace} class provides similar
 * functionality at the higher level of abstraction. Therefore, we recomend
 * to monitor this class when you want to be informed about changes in concrete human face.
 * <p>
 * Events fired by the class:
 * <ul>
 * <li>{@link cz.fidentis.analyst.mesh.events.FacetAddedEvent} when new facet is added.</li>
 * </ul>
 * </p>
 * 
 * @author Matej Lukes
 * @author Radek Oslejsek
 */
public class MeshModel {
    
    private final List<MeshFacet> facets = new ArrayList<>();
    
    private EventBus eventBus = new EventBus();
    
    /**
     * Visitor stored temporarily for later cuncurrent exploration of the mesh.
     */
    private MeshVisitor visitor;
    
    /**
     * Constructor of MeshModel
     */
    public MeshModel() {

    }

    /**
     * Copy constructor of MeshModel
     *
     * @param meshModel copied MeshModel
     */
    public MeshModel(MeshModel meshModel) {
        for (MeshFacet facet: meshModel.facets) {
            facets.add(new MeshFacetImpl(facet));
        }
    }

    /**
     * returns list of MeshFacets
     *
     * @return list of MeshFacets
     */
    public List<MeshFacet> getFacets() {
        return Collections.unmodifiableList(facets);
    }

    /**
     * Adds a new mesh facet to the model. 
     * Fires {@link cz.fidentis.analyst.mesh.events.FacetAddedEvent} to
     * registered listeners.
     *
     * @param facet new MeshFacet
     */
    public void addFacet(MeshFacet facet) {
        facets.add(facet);
        eventBus.post(new FacetAddedEvent(facet));
    }
    
    /**
     * Applies the visitor to all mesh facets, either sequentially or in parallel.
     * 
     * @param visitor Visitor to be apllied for the computation
     * @param exec Exisitng executor. If {@code null}, then a new private executor is used.
     * @throws {@code NullPointerException} if the visitor is {@code null}
     */
    public void compute(MeshVisitor visitor, ExecutorService exec) {
        if (visitor.concurrently()) {
            ExecutorService executor = exec;
            if (executor == null) {
                int threads = Runtime.getRuntime().availableProcessors();
                executor = Executors.newFixedThreadPool(threads);
            }
            
            List<Future<MeshVisitor>> results = new LinkedList<>();
            
            for (MeshFacet f: this.facets) {
                f.accept(visitor);
                Future<MeshVisitor> result = executor.submit(visitor); // fork and continue
                results.add(result);
            }
            
            //executor.shutdown();
            
            try {
                for (Future<MeshVisitor> f : results) { 
                    f.get(); // wait ntil all computations are finished
                }
            } catch (InterruptedException | ExecutionException ex) {
                Logger.getLogger(MeshModel.class.getName()).log(Level.SEVERE, null, ex);
            }
        } else {
            for (MeshFacet f: this.facets) {
                f.accept(visitor);
            }
        }
    }
    
    /**
     * Applies the visitor to all mesh facets, either sequentially or in parallel.
     * 
     * @param visitor Visitor to be apllied
     */
    public void compute(MeshVisitor visitor) {
        compute(visitor, null);
    }
    
    /**
     * Registers listeners (objects concerned in the mesh model changes) to receive events.
     * If listener is {@code null}, no exception is thrown and no action is taken.
     * 
     * @param listener Listener concerned in the mesh model changes.
     */
    public void registerListener(MeshListener listener) {
        eventBus.register(listener);
    }
    
    /**
     * Unregisters listeners from receiving events.
     * 
     * @param listener Registered listener
     */
    public void unregisterListener(MeshListener listener) {
        eventBus.unregister(listener);
    }
}
