package cz.fidentis.analyst.mesh;

import cz.fidentis.analyst.mesh.core.MeshFacet;
import java.util.concurrent.Callable;

/**
 * A functor. When instantiated, it can be gradually applied to multiple mesh facets.
 * It inspects the state of the mesh facets one by one, and (cumulatevely) computes results.
 * <p>
 * Implement this interface whenever you want to define new algorithm over a mesh.
 * </p>
 * <p>
 * The visitor can be instatiated either as sequential or concurrent. 
 * The {@link cz.fidentis.analyst.mesh.MeshVisitor#visit} inspection method
 * of a sequential visitor is applied immediately. On the contrary, if the
 * visitor is concurrent and thread-safe, then the methed only stores the inpected
 * mesh facet for later concurrent inspection. To trigger the facet inspection,
 * a {@link java.util.concurrent.ExecutorService} has to be used to call
 * the {@link cz.fidentis.analyst.mesh.mesh.MeshVisitor#call} method asynchronously
 * and then waiting for the results of all triggered visitors. 
 * </p>
 * <p>
 * All visitors have to be cloneable. A returned clone represents a visitor in 
 * the initial state, i.e., only the configuration options provided to the constructor 
 * are cloned while the computation state is set to initial values.
 * </p>
 * 
 * @author Radek Oslejsek
 */
public abstract class MeshVisitor implements Callable<MeshVisitor>, Cloneable {
    
    private boolean concurrently;
    
    /**
     * Facet stored temporarily for later cuncurrent inspection via 
     * the {@link cz.fidentis.analyst.mesh.MeshVisitor#call} method.
     */
    private MeshFacet facet;
    
    
    /**
     * Constructor.
     * 
     * @param concurrently If {@code true} and the visitor is thread-safe, then 
     * the visitor is created as concurent
     * (the {@link cz.fidentis.analyst.mesh.MeshVisitor#visit} is not executed 
     * immediately but postponed for concurrent asynchronous execution). 
     * Otherwise, the visitor is created as sequential 
     * (the {@link cz.fidentis.analyst.mesh.MeshVisitor#visit} is executed immediately.)
     */
    public MeshVisitor(boolean concurrently) {
        this.concurrently = concurrently;
    }
    
    /**
     * Returns {@code true} if the implementation is thread-safe and then
     * <b>a single visitor instance</b> can be applied to multiple mesh facets simultaneously.
     * If the visitor is not thread-safe, the concurrent inspection 
     * via the {@link cz.fidentis.analyst.mesh.MeshVisitor#visit}
     * is still possible, but with multiple visitor instances - one for each facet.
     * <p>
     * Thread-safe implementation means that any read or write from/to the visitor's 
     * state is implemented as {@code synchronized}.
     * </p>
     * 
     * @return {@code true} if the implementation is thread-safe.
     */
    public boolean isThreadSafe() {
        return true;
    }
    
    /**
     * Returns {@code true} if this visitor is to be applied concurrently, i.e.,
     * if the concurrency parameter was set and the visitor is thread-safe.
     * 
     * @return {@code true} if this visitor is to be applied concurrently, i.e.,
     * if the concurrency parameter was set and the visitor is thread-safe.
     */
    public boolean concurrently() {
        return this.concurrently && isThreadSafe();
    }
    
    /**
     * The inspection method to be implememted by specific visitors.
     * 
     * @param facet Mesh facet to be visited.
     */
    protected abstract void visitMeshFacet(MeshFacet facet);
    
    /**
     * The main visitation method, either invoked immediately (if the visitor 
     * was created a sequentional) or postponed to later parallel execution 
     * (if the visitor is thread-safe and has been created a concurrent).
     * 
     * @param facet Mesh facet to be visited.
     */
    public void visit(MeshFacet facet) {
        if (concurrently && isThreadSafe()) {
            this.facet = facet;
        } else {
            visitMeshFacet(facet);
        }
    }
    
    @Override
    public MeshVisitor call() throws Exception {
        visitMeshFacet(facet);
        return this;
    }

    /**
     * Creates and returns a copy of this visitor. A returned clone represents a visitor in 
     * the initial state, i.e., only the configuration options provided to the constructor 
     * are cloned while the computation state is set to initial values. 
     * 
     * @return
     */
    @Override
    public abstract MeshVisitor clone();
}
