package cz.fidentis.analyst.kdtree;

import java.util.concurrent.Callable;

/**
 * A functor. When instantiated, it can be gradually applied to multiple k-d trees.
 * It inspects the state of the tree one by one, and (cumulatevely) computes results.
 * <p>
 * Implement this interface whenever you want to define new algorithm over a k-d tree.
 * </p>
 * <p>
 * The visitor can be instatiated either as sequential and concurrent. 
 * The {@link cz.fidentis.analyst.kdtree.KdTreeVisitor#visit} inspection method
 * of a sequential visitor is applied immediately. On the contrary, if the
 * visitor is concurrent and thread-safe, then the method only stores the inpected
 * k-d tree for later concurrent inspection. To trigger the tree inspection,
 * a {@link java.util.concurrent.ExecutorService} has to be used calling
 * the {@link cz.fidentis.analyst.mesh.mesh.MeshVisitor#call} method asynchronously
 * and then waiting for the results of all triggered visitors. 
 * </p>
 * <p>
 * All visitors have to be cloneable. A returned clone represents a visitor in 
 * the initial state, i.e., only the configuration options provided to the constructor 
 * are cloned while the computation state is set to initial values.
 * </p>
 *
 * @author Daniel Schramm
 */
public abstract class KdTreeVisitor implements Callable<KdTreeVisitor>, Cloneable {
    
    private final boolean concurrently;
    
    /**
     * KD tree stored temporarily for later cuncurrent inspection via 
     * the {@link cz.fidentis.analyst.kdtree.KdTreeVisitor#visit} method.
     */
    private KdTree kdTree;

    /**
     * Constructor.
     * 
     * @param concurrently If {@code true} and the visitor is thread-safe, then 
     * the visitor is created as concurent
     * (the {@link cz.fidentis.analyst.kdtree.KdTreeVisitor#visit} is not executed 
     * immediately but postponed for concurrent asynchronous execution). 
     * Otherwise, the visitor is created as sequential 
     * (the {@link cz.fidentis.analyst.kdtree.KdTreeVisitor#visit} is executed immediately.)
     */
    public KdTreeVisitor(boolean concurrently) {
        this.concurrently = concurrently;
    }
    
    /**
     * Copy constructor.
     * 
     * @param vis source visitor to be coppied
     * @throws NullPointerException if the input parameter is {@code null}
     */
    protected KdTreeVisitor(KdTreeVisitor vis) {
        concurrently = vis.concurrently;
    }
    
    /**
     * Returns {@code true} if the implementation is thread-safe and then
     * <b>a single visitor instance</b> can be applied to multiple KD trees simultaneously.
     * If the visitor is not thread-safe, the concurrent inspection 
     * via the {@link cz.fidentis.analyst.kdtree.KdTreeVisitor#visit}
     * is still possible, but with multiple visitor instances - one for each tree.
     * <p>
     * Thread-safe implementation means that any read or write from/to the visitor's 
     * state is implemented as {@code synchronized}.
     * </p>
     * 
     * @return {@code true} if the implementation is thread-safe.
     */
    public boolean isThreadSafe() {
        return true;
    }
    
    /**
     * Returns {@code true} if this visitor is to be applied concurrently, i.e.,
     * if the concurrency parameter was set and the visitor is thread-safe.
     * 
     * @return {@code true} if this visitor is to be applied concurrently, i.e.,
     * if the concurrency parameter was set and the visitor is thread-safe.
     */
    public boolean concurrently() {
        return this.concurrently && isThreadSafe();
    }
    
    /**
     * The inspection method to be implememted by specific visitors.
     * 
     * @param kdTree KD tree to be visited
     */
    protected abstract void visitKdTree(KdTree kdTree);
    
    /**
     * The main visitation method, either invoked immediately (if the visitor 
     * was created a sequentional) or postponed to later parallel execution 
     * (if the visitor is thread-safe and has been created a concurrent).
     * 
     * @param kdTree KD tree to be visited
     */
    public void visit(KdTree kdTree) {
        if (concurrently && isThreadSafe()) {
            this.kdTree = kdTree;
        } else {
            visitKdTree(kdTree);
        }
    }
    
    @Override
    public KdTreeVisitor call() throws Exception {
        visitKdTree(kdTree);
        return this;
    }
    
    /**
     * Creates and returns a copy of this visitor. A returned clone represents a visitor in 
     * the initial state, i.e., only the configuration options provided to the constructor 
     * are cloned while the computation state is set to initial values. 
     * 
     * @return a copy of this visitor
     */
    @Override
    public abstract Object clone();
}
