package cz.fidentis.analyst.tests;

import cz.fidentis.analyst.batch.Stopwatch;
import cz.fidentis.analyst.face.HumanFace;
import cz.fidentis.analyst.face.HumanFaceFactory;
import cz.fidentis.analyst.mesh.core.MeshModel;
import cz.fidentis.analyst.mesh.io.MeshObjExporter;
import cz.fidentis.analyst.avgmesh.AvgMeshConstructorOctree;
import java.io.File;
import java.io.IOException;

/**
 * A class for testing the correctness of batch Ray tracing algorithm. Algorithm
 * is used on two faces.
 *
 * @author Enkh-Undral EnkhBayar
 */
public class BatchOctreeTestsTwoFaces {

    private static final String PATH_MAIN = "/home/uenkhbayar/git/fidentis/mainFaceScaled.obj";
    private static final String PATH_SECONDARY = "/home/uenkhbayar/git/fidentis/secondFaceScaled.obj";
//    private static final String pathMain = "/home/uenkhbayar/git/fidentis/analyst-data/basic/01.obj";
//    private static final String pathsecondary = "/home/uenkhbayar/git/fidentis/analyst-data/basic/02.obj";

    private static final String PATH_MAIN_ALIGNED = "/home/uenkhbayar/git/fidentis/mainFace.obj";
    private static final String PATH_SECONDARY_ALIGNED = "/home/uenkhbayar/git/fidentis/secondFaceScaled.obj";

    private static final String PATH_AVG_FACE = "/home/uenkhbayar/git/fidentis/avgFace.obj";

    private static final Stopwatch TOTAL_TIME = new Stopwatch("Total computation time:\t");
    private static final Stopwatch AVG_FACE_COMPUTATION_TIME = new Stopwatch("AVG face computation time:\t");
    private static final Stopwatch ICP_COMPUTATION_TIME = new Stopwatch("ICP registration time:\t");
    private static final Stopwatch LOAD_TIME = new Stopwatch("File (re-)loading time:\t");
    private static final Stopwatch OCTREE_CONSTRUCTION_TIME = new Stopwatch("Octree construction time:\t");

    /**
     * 
     * @param args
     * @throws IOException 
     */
    public static void main(String[] args) throws IOException {
        HumanFaceFactory factory = new HumanFaceFactory();

        File mainFile = new File(PATH_MAIN);
        File secondFile = new File(PATH_SECONDARY);

        factory.setReuseDumpFile(true);
        factory.setStrategy(HumanFaceFactory.Strategy.MRU);

        TOTAL_TIME.start();

        LOAD_TIME.start();
        String mainFaceId = factory.loadFace(mainFile);
        HumanFace mainFace = factory.getFace(mainFaceId);
        LOAD_TIME.stop();

        LOAD_TIME.start();
        String secondFaceId = factory.loadFace(secondFile);
        HumanFace secondFace = factory.getFace(secondFaceId);
        LOAD_TIME.stop();

//        icpComputationTime.start();
//        HumanFaceUtils.alignMeshes(mainFace,
//                secondFace, // is transformed
//                100, // max iterations
//                true,
//                0.3, // error
//                new RandomSampling(100),
//                false // drop k-d tree, if exists
//        );
//        icpComputationTime.stop();
//        writeToFile(mainFace.getMeshModel(), pathMainAligned);
//        writeToFile(secondFace.getMeshModel(), pathSecondaryAligned);
//        printVerticesCount(mainFace);

        OCTREE_CONSTRUCTION_TIME.start();
        secondFace.computeOctree(false);
        OCTREE_CONSTRUCTION_TIME.stop();

        AVG_FACE_COMPUTATION_TIME.start();
        var avgFaceConstructorOctree = new AvgMeshConstructorOctree(mainFace.getMeshModel());
        avgFaceConstructorOctree.visitOctree(secondFace.getOctree());
        AVG_FACE_COMPUTATION_TIME.stop();

        writeToFile(avgFaceConstructorOctree.getAveragedMeshModel(), PATH_AVG_FACE);

        TOTAL_TIME.stop();

//        printTimeStats();
    }

    protected static void printVerticesCount(HumanFace face) {
        int count = 0;
        for (var facet : face.getMeshModel().getFacets()) {
            count += facet.getNumberOfVertices();
        }
        System.out.println("count of vertices: " + count);
    }

    protected static void writeToFile(MeshModel model, String path) {
        File file = new File(path);
        try {
            new MeshObjExporter(model).exportModelToObj(file);
        } catch (IOException ex) {
            System.err.println(ex.toString());
        }
    }

    protected static void printTimeStats() {
        System.out.println(AVG_FACE_COMPUTATION_TIME.toString());
        System.out.println(ICP_COMPUTATION_TIME.toString());
        System.out.println(LOAD_TIME.toString());
        System.out.println(OCTREE_CONSTRUCTION_TIME.toString());
        System.out.println(TOTAL_TIME.toString());
    }

}
