package cz.fidentis.analyst.symmetry;

import cz.fidentis.analyst.shapes.CrossSectionCurve;
import cz.fidentis.analyst.drawables.DrawableFace;
import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Stroke;
import java.awt.geom.Ellipse2D;
import java.awt.geom.Line2D;
import java.util.List;
import javax.swing.JPanel;
import javax.vecmath.Point3d;


/**
 * Panel for graphing polyline profiles
 *
 * @author Dominik Racek
 * @author Radek Oslejsek
 */
public class CurveRenderingPanel extends JPanel {
    
    private static final int SEGMENT_POINT_SIZE = 7;
    private static final int PREF_W = 400;
    private static final int PREF_H = 400;
    private static final int BORDER_GAP = 15;
    private static final Color PRIMARY_COLOR = DrawableFace.SKIN_COLOR_PRIMARY;
    private static final Color PRIMARY_MIRROR_COLOR = PRIMARY_COLOR.darker();
    private static final Color SECONDARY_COLOR = DrawableFace.SKIN_COLOR_SECONDARY.darker();
    private static final Color SECONDARY_MIRROR_COLOR = SECONDARY_COLOR.darker().darker();
    /*
    private static final Color PRIMARY_COLOR = Color.green;
    private static final Color PRIMARY_MIRROR_COLOR = new Color(0, 85, 0);
    private static final Color SECONDARY_COLOR = new Color(0, 155, 200);
    private static final Color SECONDARY_MIRROR_COLOR = Color.blue;
    */
    private static final Stroke GRAPH_STROKE = new BasicStroke(3f);
    
    private CrossSectionCurve primarySegments;
    private CrossSectionCurve secondarySegments;
    private CrossSectionCurve primaryMirrorSegments;
    private CrossSectionCurve secondaryMirrorSegments;
    
    private boolean mirrorCuts = false;
    
    private double minZ = Double.POSITIVE_INFINITY;
    private double maxZ = Double.NEGATIVE_INFINITY;
    private double minY = Double.POSITIVE_INFINITY;
    private double maxY = Double.NEGATIVE_INFINITY;
    private double scale = Double.POSITIVE_INFINITY;
    
    /**
     * Constructor.
     */
    public CurveRenderingPanel() {
        setPreferredSize(new Dimension(PREF_W, PREF_H));
    }

    /**
     * Sets primary points and draws them in the panel
     *
     * @param curve Points of the primary curve
     */
    public void setPrimarySegments(CrossSectionCurve curve) {
        this.primarySegments = curve;
        updateMinMax(curve);
        repaint();
    }

    /**
     * Sets primary mirror points.
     * Only draws them if mirrorCuts is checked
     *
     * @param curve Points of the primary mirror curve
     */
    public void setPrimaryMirrorSegments(CrossSectionCurve curve) {
        this.primaryMirrorSegments = curve;
        updateMinMax(curve);
        repaint();
    }

    /**
     * Sets secondary points and draws them in the panel
     *
     * @param curve Points of the secondary curve
     */
    public void setSecondarySegments(CrossSectionCurve curve) {
        this.secondarySegments = curve;
        updateMinMax(curve);
        repaint();
    }

    /**
     * Sets secondary mirror points.
     * Only draws them if mirrorCuts is checked
     *
     * @param curve Points of the secondary mirror curve
     */
    public void setSecondaryMirrorSegments(CrossSectionCurve curve) {
        this.secondaryMirrorSegments = curve;
        updateMinMax(curve);
        repaint();
    }

    /**
     * Displays the mirror cuts
     *
     * @param mirror
     */
    public void setMirrorCuts(boolean mirror) {
        this.mirrorCuts = mirror;
        repaint();
    }

    @Override
    public Dimension getPreferredSize() {
        return new Dimension(PREF_W, PREF_H);
    }

    @Override
    protected void paintComponent(Graphics g) {
        super.paintComponent(g);
        Graphics2D g2 = (Graphics2D)g;
        
        if (mirrorCuts && secondarySegments != null) { // two faces with mirror cutting planes => double view
            double offsetZ = (PREF_W * 0.5) - (maxZ * scale);
            drawFace(g2, primarySegments, PRIMARY_COLOR, offsetZ);
            drawFace(g2, secondarySegments, SECONDARY_COLOR, offsetZ);
            offsetZ = (PREF_W * 1.0) - (maxZ * scale);
            drawFace(g2, primaryMirrorSegments, PRIMARY_COLOR, offsetZ);
            drawFace(g2, secondaryMirrorSegments, SECONDARY_COLOR, offsetZ);
        } else { // either a single face or two faces without mirror cutting planes => single view
            double offsetZ = (PREF_W * 0.7) - (maxZ * scale);
            drawFace(g2, primarySegments, PRIMARY_COLOR, offsetZ);
            if (mirrorCuts) { // single face with mirror cutting plane
                drawFace(g2, primaryMirrorSegments, PRIMARY_COLOR.darker(), offsetZ);
            } else { // two faces without mirror cutting plane
                drawFace(g2, secondarySegments, SECONDARY_COLOR, offsetZ);
            }
        }
    }

    protected void drawFace(Graphics2D g2, CrossSectionCurve curve, Color faceColor, double offsetZ) {
        if (curve == null) {
            return;
        }
        
        for (int i = 0; i < curve.getNumSegments(); i++) {
            drawCurveSegment(g2, curve.getSegment(i), faceColor, offsetZ);
        }
        drawProjectedFeaturePoints(g2, curve.getProjectedFeaturePoints(), faceColor, offsetZ);
    }

    protected void drawCurveSegment(Graphics2D g2, List<Point3d> curveSegment, Color faceColor, double offsetZ) {
        //Draw lines
        g2.setColor(faceColor);
        g2.setStroke(GRAPH_STROKE);
        for (int i = 0; i < curveSegment.size() - 1; i++) {
            double z1 = curveSegment.get(i).z * scale + BORDER_GAP + offsetZ;
            //double y1 = (maxY - curveSegment.get(i).y) * scale + BORDER_GAP;
            double y1 = PREF_H / 2 - curveSegment.get(i).y * scale + 2 * BORDER_GAP;
            double z2 = (curveSegment.get(i + 1).z) * scale + BORDER_GAP + offsetZ;
            //double y2 = (maxY - curveSegment.get(i + 1).y) * scale + BORDER_GAP;
            double y2 = PREF_H / 2 - curveSegment.get(i + 1).y * scale + 2 * BORDER_GAP;
            
            g2.draw(new Line2D.Double(z1, y1, z2, y2));
        }
    }
    
    protected void drawProjectedFeaturePoints(Graphics2D g2, List<Point3d> fps, Color faceColor, double offsetZ) {
        g2.setColor(faceColor);
        for (int i = 0; i < fps.size(); i++) {
            double z = fps.get(i).z * scale + BORDER_GAP + offsetZ;
            double y = PREF_H / 2 - fps.get(i).y * scale + 2 * BORDER_GAP;
            Ellipse2D.Double point = new Ellipse2D.Double(
                    z - SEGMENT_POINT_SIZE / 2d, 
                    y - SEGMENT_POINT_SIZE / 2d, 
                    SEGMENT_POINT_SIZE, 
                    SEGMENT_POINT_SIZE);
            g2.fill(point);
            g2.draw(point);
        }
    }

    protected void updateMinMax(CrossSectionCurve curve) {
        if (curve == null) {
            return;
        }
        
        for (int i = 0; i < curve.getNumSegments(); i++) {
            for (int j = 0; j < curve.getSegment(i).size(); j++) {
                minZ = Math.min(minZ, curve.getSegment(i).get(j).z);
                maxZ = Math.max(maxZ, curve.getSegment(i).get(j).z);
                minY = Math.min(minY, curve.getSegment(i).get(j).y);
                maxY = Math.max(maxY, curve.getSegment(i).get(j).y);
            }
        }

        scale = ((double) PREF_H - 2 * BORDER_GAP) / (maxY - minY);
        scale *= 0.9;
    }
}