package cz.fidentis.analyst.project;

import cz.fidentis.analyst.face.HumanFace;
import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.prefs.Preferences;
import com.fasterxml.jackson.databind.ObjectMapper;
import cz.fidentis.analyst.Logger;
import java.nio.file.Paths;


/**
 * This class encapsulates data for a "comparison project", i.e., a project 
 * enabling analysts to compare and investigate human faces.
 *
 * @author Matej Kovar
 */
public class Project {
    
    private boolean saved = true;
    private String projectName = "";
    
    private ObjectMapper mapper = new ObjectMapper();
    
    private Preferences userPreferences = Preferences.userNodeForPackage(Project.class); 
    
    /* Project data (paths to faces) */
    private ProjectConfiguration cfg = new ProjectConfiguration();
    
    /**
     * Asks whether project is saved or some change was made
     * @return true if project is saved, false otherwise
     */
    public boolean isSaved() {
        return saved;
    }

    public void setSaved(boolean saved) {
        this.saved = saved;
    }

    public ProjectConfiguration getCfg() {
        return cfg;
    }

    public void setCfg(ProjectConfiguration cfg) {
        this.cfg = cfg;
    }
    
    public String getProjectName() {
        return projectName;
    }
    
    /**
     * Adds new path to project configuration
     * @param path Path to be added
     * @return true if path was successfully added
     */
    public boolean addNewPath(Path path) {
        return this.cfg.addPath(path);
    }
    
    /**
     * Checks whether path is already loaded in project configuration
     * @param path Path to be checked
     * @return true if path was loaded, false otherwise
     */
    public boolean pathLoaded(Path path) {
        return this.cfg.getPaths().contains(path);
    }
    
    /**
     * Loads face from path
     * @param name String name of face
     * @return loaded HumanFace or null
     */
    public HumanFace loadFace(String name) {
        try {
            Logger out = Logger.measureTime();
            Path path = this.getCfg().getPathToFaceByName(name);
            File file = path.toFile();
            HumanFace face = new HumanFace(file, true); // loads also landmarks, if exist
            out.printDuration("Loaded model " + face.getShortName() +" with " + face.getMeshModel().getNumVertices() + " vertices");
            return face;
        } catch (IOException ex) {
            //ex.printStackTrace();
            Logger.print(ex.toString());
        }    
        return null;
    }
    
    /**
     * Checks whether model saved in this project has in the same directory
     * file with feature points
     * @param name String name of face
     * @return true if feature points file was found, false otherwise
     */
    public boolean hasFPByName(String name) {
        
        Path path = this.getCfg().getPathToFaceByName(name);
        
        if (path != null) {
            
            String filePath = path.toString().split(".obj")[0] + "_landmarks.csv";
            if ((new File(filePath)).exists()) {
                return true;
            }

            filePath = path.toString().split("_ECA")[0] + "_landmarks.csv";
            if ((new File(filePath)).exists()) {
                return true;
            }

            filePath = path.toString().split("_CA")[0] + "_landmarks.csv";
            if ((new File(filePath)).exists()) {
                return true;
            }
        }
        return false;
    }
    
    /**
     * Close currently loaded project
     */
    public void closeProject() {
        userPreferences.remove("pathToMostRecentProject");
    }
    
    /**
     * Save currently loaded project
     * @param file file where to store project
     * @return true if project was successfully saved
     */
    public boolean saveProject(File file) {
        
        try {
            mapper.writeValue(file, cfg);
            userPreferences.put("pathToMostRecentProject", file.getAbsolutePath());
            setSaved(true);
            projectName = file.getName().split(".json")[0];
            return true;
        } catch (IOException ex) {
            //Exceptions.printStackTrace(ex);
        }
        return false;
    }
    
    /**
     * Create new project
     */
    public void newProject() {
        setCfg(new ProjectConfiguration());
        projectName = "";
        setSaved(true);
    }
    
    /**
     * Gets path to project file
     * @return String path to project file
     */
    public String getPathToProjectFile() {
        return userPreferences.get("pathToMostRecentProject", "");
    }
    
    /**
     * Loads most recent project from user preferences
     * @return File where project is saved, null if no project was found
     */
    public File getRecentProject() {
        
        String path = getPathToProjectFile();
        if (path.equals("")) {
            return null;
        }
        Path p = Paths.get(path);
        return p.toFile();
    }
   
    /**
     * Loads project from file
     * @param projectFile file with project
     * @return true if project was successfully loaded, false otherwise
     */
    public boolean openProjectFromFile(File projectFile) {
        if (projectFile != null) {
            try {
                closeProject();
                cfg = mapper.readValue(projectFile, ProjectConfiguration.class);
                userPreferences.put("pathToMostRecentProject", projectFile.getAbsolutePath());
                setSaved(true);
                projectName = projectFile.getName().split(".json")[0];
            } catch (IOException ex) {
                return false;
            }
        } 
        return true;
    }
}
