/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package cz.fidentis.analyst.gui;

import com.jogamp.opengl.GL2;
import cz.fidentis.analyst.gui.canvas.Canvas;
import cz.fidentis.analyst.gui.scene.DrawableMesh;
import cz.fidentis.analyst.mesh.core.MeshFacet;
import cz.fidentis.analyst.mesh.core.MeshPoint;
import cz.fidentis.analyst.visitors.mesh.BoundingBox;
import java.awt.Color;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Collections;
import javax.vecmath.Point3d;
import javax.vecmath.Vector3d;

/**
 * Executes changes made in PostRegistrationCP
 * 
 * @author Richard Pajersky
 */
public class RegistrationCPEventListener {
    
    private final double featurePointsThreshold = 0.1;
    private final Color defaultPrimaryColor = new Color(51, 51, 153);
    private final Color defaultSecondaryColor = new Color(255, 239, 0);
    private final int transparencyRange = 10;
    private final double changeAmount = 500d;
    private final Canvas canvas;
    private final double moveX;
    private final double moveY;
    private final double moveZ;
    private final double scaleXYZ;
    
    private DrawableMesh primaryFace;
    private DrawableMesh secondaryFace;
    
    public RegistrationCPEventListener(Canvas canvas) {
        this.canvas = canvas;
        
        ArrayList<DrawableMesh> drawables = new ArrayList<>(canvas.getScene().getDrawables());
        primaryFace = drawables.get(0);
        secondaryFace = drawables.get(1);
        setDeafultColor();
        
        // set change amounts
        BoundingBox visitor = new BoundingBox();
        secondaryFace.getModel().compute(visitor);
        Point3d maxPoint = visitor.getBoundingBox().getMaxPoint();
        Point3d minPoint = visitor.getBoundingBox().getMinPoint();
        moveX = (maxPoint.x - minPoint.x) / changeAmount;
        moveY = (maxPoint.y - minPoint.y) / changeAmount;
        moveZ = (maxPoint.z - minPoint.z) / changeAmount;
        scaleXYZ = (visitor.getBoundingBox().getMaxDiag() / (10 * changeAmount));
    }
    
    /**
     * Calculates feature points which are too far away
     */
    private void calculateFeaturePoints() {
        if (!primaryFace.isRenderFeaturePoints()) {
            return;
        }
        ArrayList<AbstractMap.SimpleEntry<Point3d, Color>> adjusted = new ArrayList<>();
        for (int i = 0; i < primaryFace.getFeaturePoints().size(); i++) {
            Point3d primaryPoint = primaryFace.getFeaturePoints().get(i).getKey();
            Point3d secondaryPoint = new Point3d(secondaryFace.getFeaturePoints().get(i).getKey());
            transformPoint(secondaryPoint);
            double distance = Math.sqrt(
                Math.pow(secondaryPoint.x - primaryPoint.x, 2) + 
                Math.pow(secondaryPoint.y - primaryPoint.y, 2) + 
                Math.pow(secondaryPoint.z - primaryPoint.z, 2));
            Point3d point = new Point3d(secondaryFace.getFeaturePoints().get(i).getKey());
            if (distance > featurePointsThreshold) {
                adjusted.add(new AbstractMap.SimpleEntry<>(
                        point, Color.RED));
            } else {
                adjusted.add(new AbstractMap.SimpleEntry<>(
                        point, Color.YELLOW));
            }
            secondaryFace.setFeaturePoints(adjusted);
        }
    }
    
    /**
     * Applies carried out transformations
     */
    public void transformFace() {
        for (MeshFacet transformedFacet : secondaryFace.getFacets()) {
            for (MeshPoint comparedPoint : transformedFacet.getVertices()) {
                transformPoint(comparedPoint.getPosition());
            }
        }
        for (var point : secondaryFace.getFeaturePoints()) {
            transformPoint(point.getKey());
        }
        canvas.renderScene();
    }
    
    /**
     * Transforms point based on transformation info from secondary face
     * 
     * @param point Point to transform
     */
    public void transformPoint(Point3d point) {
        Point3d newPoint = new Point3d(0, 0, 0);
        double quotient;

        // rotate around X
        quotient = Math.toRadians(secondaryFace.getRotation().x);
        if (!Double.isNaN(quotient)) {
            double cos = Math.cos(quotient);
            double sin = Math.sin(quotient);
            newPoint.y = point.y * cos - point.z * sin;
            newPoint.z = point.z * cos + point.y * sin;
            point.y = newPoint.y;
            point.z = newPoint.z;
        }

        // rotate around Y
        quotient = Math.toRadians(secondaryFace.getRotation().y);
        if (!Double.isNaN(quotient)) {
            double cos = Math.cos(quotient);
            double sin = Math.sin(quotient);
            newPoint.x = point.x * cos + point.z * sin;
            newPoint.z = point.z * cos - point.x * sin;
            point.x = newPoint.x;
            point.z = newPoint.z;
        }

        // rotate around Z
        quotient = Math.toRadians(secondaryFace.getRotation().z);
        if (!Double.isNaN(quotient)) {
            double cos = Math.cos(quotient);
            double sin = Math.sin(quotient);
            newPoint.x = point.x * cos - point.y * sin;
            newPoint.y = point.y * cos + point.x * sin;
            point.x = newPoint.x;
            point.y = newPoint.y;
        }

        // translate
        point.x += secondaryFace.getTranslation().x;
        point.y += secondaryFace.getTranslation().y;
        point.z += secondaryFace.getTranslation().z;

        // scale
        point.x *= 1 + secondaryFace.getScale().x;
        point.y *= 1 + secondaryFace.getScale().y;
        point.z *= 1 + secondaryFace.getScale().z;
    }
    
    public final void setDeafultColor() {
        primaryFace.setColor(defaultPrimaryColor);
        secondaryFace.setColor(defaultSecondaryColor);
    }
    
    public int getTransparencyRange() {
        return transparencyRange;
    }
    
    public void setPrimaryColor(Color color) {
        primaryFace.setColor(color);
        canvas.renderScene();
    }

    public void setSecondaryColor(Color color) {
        secondaryFace.setColor(color);
        canvas.renderScene();
    }
    
    public Color getPrimaryColor() {
        return primaryFace.getColor();
    }

    public Color getSecondaryColor() {
        return secondaryFace.getColor();
    }
    
    public void setTransparency(int value) {
        
        if (value == transparencyRange) {
            setPrimaryTransparency(1);
            setSecondaryTransparency(1);
        }
        if (value < transparencyRange) {
            setPrimaryTransparency(value / 10f);
            setSecondaryTransparency(1);
        }
        if (value > transparencyRange) {
            setSecondaryTransparency((2 * transparencyRange - value) / 10f);
            setPrimaryTransparency(1);
        }
        canvas.renderScene();
    }
    
    public void setPrimaryTransparency(float transparency) {
        primaryFace.setTransparency(transparency);
    }

    public void setSecondaryTransparency(float transparency) {
        secondaryFace.setTransparency(transparency);
    }
    
    public double getPrimaryTransparency() {
        return primaryFace.getTransparency();
    }

    public double getSecondaryTransparency() {
        return secondaryFace.getTransparency();
    }
    
    public void setFrontFacing() {
        canvas.getCamera().initLocation();
        canvas.renderScene();
    }
    
    public void setSideFacing() {
        canvas.getCamera().initLocation();
        canvas.getCamera().rotate(0, 90);
        canvas.renderScene();
    }
    
    public Color getDefaultPrimaryColor() {
        return defaultPrimaryColor;
    }

    public Color getDefaultSecondaryColor() {
        return defaultSecondaryColor;
    }
    
    public void resetTranslation() {
        secondaryFace.setTranslation(new Vector3d(0, 0, 0));
        calculateFeaturePoints();
        canvas.renderScene();
    }
    
    public void resetRotation() {
        secondaryFace.setRotation(new Vector3d(0, 0, 0));
        calculateFeaturePoints();
        canvas.renderScene();
    }

    public void resetScale() {
        secondaryFace.setScale(new Vector3d(0, 0, 0));
        calculateFeaturePoints();
        canvas.renderScene();
    }
    
    public void setXTranslation(double value) {
        secondaryFace.getTranslation().x = value * moveX;
        calculateFeaturePoints();
        canvas.renderScene();
    }
    
    public void setYTranslation(double value) {
        secondaryFace.getTranslation().y = value * moveY;
        calculateFeaturePoints();
        canvas.renderScene();
    }
        
    public void setZTranslation(double value) {
        secondaryFace.getTranslation().z = value * moveZ;
        calculateFeaturePoints();
        canvas.renderScene();
    }
    
    public void setXRotation(double value) {
        secondaryFace.getRotation().x = value * moveX;
        calculateFeaturePoints();
        canvas.renderScene();
    }
    
    public void setYRotation(double value) {
        secondaryFace.getRotation().y = value * moveY;
        calculateFeaturePoints();
        canvas.renderScene();
    }
        
    public void setZRotation(double value) {
        secondaryFace.getRotation().z = value * moveZ;
        calculateFeaturePoints();
        canvas.renderScene();
    }
    
    public void setScale(double value) {
        secondaryFace.getScale().x = value * scaleXYZ;
        secondaryFace.getScale().y = value * scaleXYZ;
        secondaryFace.getScale().z = value * scaleXYZ;
        calculateFeaturePoints();
        canvas.renderScene();
    }
    
    public void setPrimaryHighlights() {
        primaryFace.setHighlights(new Color(1, 1, 1));
        canvas.renderScene();
    }
    
    public void setSecondaryHighlights() {
        secondaryFace.setHighlights(new Color(1, 1, 1));
        canvas.renderScene();
    }
    
    public void removePrimaryHighlights() {
        primaryFace.setHighlights(new Color(0, 0, 0));
        canvas.renderScene();
    }
    
    public void removeSecondaryHighlights() {
        secondaryFace.setHighlights(new Color(0, 0, 0));
        canvas.renderScene();
    }
    
    public void setPrimaryLines() {
        primaryFace.setRenderMode(GL2.GL_LINE);
        canvas.renderScene();
    }
    
    public void setSecondaryLines() {
        secondaryFace.setRenderMode(GL2.GL_LINE);
        canvas.renderScene();
    }
    
    public void setPrimaryPoints() {
        primaryFace.setRenderMode(GL2.GL_POINT);
        canvas.renderScene();
    }
    
    public void setSecondaryPoints() {
        secondaryFace.setRenderMode(GL2.GL_POINT);
        canvas.renderScene();
    }
    
    public void setPrimaryFill() {
        primaryFace.setRenderMode(GL2.GL_FILL);
        canvas.renderScene();
    }
    
    public void setSecondaryFill() {
        secondaryFace.setRenderMode(GL2.GL_FILL);
        canvas.renderScene();
    }
    
    public boolean isFeaturePointsActive() {
        return primaryFace.isRenderFeaturePoints();
    }
    
    public void setFeaturePointsActive(boolean state) {
        primaryFace.setRenderFeaturePoints(state);
        secondaryFace.setRenderFeaturePoints(state);
        calculateFeaturePoints();
        canvas.renderScene();
    }
}
