package cz.fidentis.analyst.mesh.core;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import javax.vecmath.Vector3d;

/**
 * Adapter for the corner table representing a single triangle of {@code MeshFacet}.
 * 
 * @author Natalia Bebjakova
 */

public class MeshTriangle implements Iterable<MeshPoint> { 
    
    public static final double EPS = 0.001; // tollerance
    
    //public final MeshPoint vertex1;
    //public final MeshPoint vertex2;
    //public final MeshPoint vertex3;
    
    private final MeshFacet facet;
    
    /**
     * Under which index is the corresponding vertex stored in the mesh facet
     */
    public final int index1;
    public final int index2;
    public final int index3;
    
    private Vector3d voronoiPoint;
    
    /**
     * Creates new triangle
     * 
     * @param v1 first vertex 
     * @param v2 second vertex
     * @param v3 third vertex
     * @param i1 which index is the first vertex stored in the mesh facet
     * @param i2 which index is the second vertex stored in the mesh facet
     * @param i3 which index is the third vertex stored in the mesh facet
     */        
    public MeshTriangle(MeshFacet facet, int i1, int i2, int i3) {
        if (facet == null) {
            throw new IllegalArgumentException("facet");
        }
        this.index1 = i1;
        this.index2 = i2;
        this.index3 = i3;
        this.facet = facet;
    }
    
    public Vector3d getVertex1() {
        return facet.getVertex(index1).getPosition();
    }
    
    public Vector3d getVertex2() {
        return facet.getVertex(index2).getPosition();
    }
    
    public Vector3d getVertex3() {
        return facet.getVertex(index3).getPosition();
    }
    
    public MeshPoint getPoint1() {
        return facet.getVertex(index1);
    }
    
    public MeshPoint getPoint2() {
        return facet.getVertex(index2);
    }
    
    public MeshPoint getPoint3() {
        return facet.getVertex(index3);
    }
    
    /**
     * Computes and returns normalized normal vector from the vertices.
     * 
     * @return normalized normal vector from the vertices.
     */
    public Vector3d computeNormal() {
        Vector3d ab = new Vector3d(getVertex1());
        ab.sub(getVertex2());
        Vector3d ac = new Vector3d(getVertex1());
        ac.sub(getVertex3());
        Vector3d normal = new Vector3d();
        normal.cross(ab, ac);
        normal.normalize();
        return normal;
    }
    
    /**
     * Computes the point laying on the triangle which is closest to 
     * given 3D point. Return point is either one of the tringle's vertices,
     * a point laying on triangles edge, or a point laying on the plane of 
     * the triangle inside the triangle boundaries.
     * 
     * @param point 3D point
     * @return the closest point or {@code null} if the input parametr is missing
     */
    public Vector3d getClosestPoint(Vector3d point) {
        if (point == null) {
            return null;
        }
        Vector3d projection = getProjectionToTrianglePlane(point);
        if (!isPointInTriangle(projection)) {
            projection = getProjectionToClosestEdge(projection);
        }
        return projection;
    }

    @Override
    public Iterator<MeshPoint> iterator() {
        return new Iterator<MeshPoint>() {
            private int counter = 0;
            
            @Override
            public boolean hasNext() {
                return counter < 3;
            }

            @Override
            public MeshPoint next() {
                if (!hasNext()) {
                    throw new NoSuchElementException();                    
                }
                switch (counter++) {
                    case 0:
                        return facet.getVertex(index1);
                    case 1:
                        return facet.getVertex(index2);
                    case 2:
                        return facet.getVertex(index3);
                    default:
                        return null;
                }                
            }
        };
    }
    
    /**
     * Return a center of circumcircle. This point represents the point
     * of Voronoi area used for Delaunay triangulation, for instenace.
     * 
     * @return the center of circumcircle
     */
    public Vector3d getVoronoiPoint() {
        if (voronoiPoint != null) {
            return voronoiPoint;
        }
        
        MeshPoint vertex1 = facet.getVertex(index1);
        MeshPoint vertex2 = facet.getVertex(index2);
        MeshPoint vertex3 = facet.getVertex(index3);
        
        double a = (vertex2.subtractPosition(vertex3)).abs();
        double b = (vertex3.subtractPosition(vertex1)).abs();
        double c = (vertex2.subtractPosition(vertex1)).abs();
        
        double d1 = a * a * (b * b + c * c - a * a);
        double d2 = b * b * (c * c + a * a - b * b);
        double d3 = c * c * (a * a + b * b - c * c);
        double dSum = d1 + d2 + d3;
        
        d1 /= dSum;
        d2 /= dSum;
        d3 /= dSum;

        MeshPoint v1Half = (vertex2.addPosition(vertex3)).dividePosition(2);
        MeshPoint v2Half = (vertex1.addPosition(vertex3)).dividePosition(2);
        MeshPoint v3Half = (vertex2.addPosition(vertex1)).dividePosition(2);

        
        if (d1 < 0) {           
            double v3Area = ((v2Half.subtractPosition(vertex3)).crossProduct(v1Half.subtractPosition(vertex3))).abs() / 2.0;
            double v2Area = ((v3Half.subtractPosition(vertex2)).crossProduct(v1Half.subtractPosition(vertex2))).abs() / 2.0;
            double v1Area = (((v1Half.subtractPosition(vertex1)).crossProduct(v3Half.subtractPosition(vertex1))).abs() / 2.0) + 
                    (((v1Half.subtractPosition(vertex1)).crossProduct(v2Half.subtractPosition(vertex1))).abs() / 2.0);
            voronoiPoint = new Vector3d(v1Area, v2Area, v3Area);
            return voronoiPoint;
        }
        if (d2 < 0) {
            double v1Area = ((v3Half.subtractPosition(vertex1)).crossProduct(v2Half.subtractPosition(vertex1))).abs() / 2.0;
            double v3Area = ((v1Half.subtractPosition(vertex3)).crossProduct(v2Half.subtractPosition(vertex3))).abs() / 2.0;
            double v2Area = (((v2Half.subtractPosition(vertex2)).crossProduct(v1Half.subtractPosition(vertex2))).abs() / 2.0) + 
                    (((v2Half.subtractPosition(vertex2)).crossProduct(v3Half.subtractPosition(vertex2))).abs() / 2.0);
            voronoiPoint = new Vector3d(v1Area, v2Area, v3Area);
            return voronoiPoint;
        }
        if (d3 < 0) {
            double v2Area = ((v1Half.subtractPosition(vertex2)).crossProduct(v3Half.subtractPosition(vertex2))).abs() / 2.0;
            double v1Area = ((v2Half.subtractPosition(vertex1)).crossProduct(v3Half.subtractPosition(vertex1))).abs() / 2.0;
            double v3Area = (((v3Half.subtractPosition(vertex3)).crossProduct(v2Half.subtractPosition(vertex3))).abs() / 2.0) + 
                    (((v3Half.subtractPosition(vertex3)).crossProduct(v1Half.subtractPosition(vertex3))).abs() / 2.0);
            voronoiPoint = new Vector3d(v1Area, v2Area, v3Area);
            return voronoiPoint;
        }
        
        MeshPoint circumcenter = vertex1.multiplyPosition(d1).addPosition(vertex2.multiplyPosition(d2).addPosition(vertex3.multiplyPosition(d3)));
        
        double v1Area = (((v2Half.subtractPosition(vertex1)).crossProduct(circumcenter.subtractPosition(vertex1))).abs() / 2.0) + 
                (((v3Half.subtractPosition(vertex1)).crossProduct(circumcenter.subtractPosition(vertex1))).abs() / 2.0);
        
        double v2Area = (((v3Half.subtractPosition(vertex2)).crossProduct(circumcenter.subtractPosition(vertex2))).abs() / 2.0) +
                (((v1Half.subtractPosition(vertex2)).crossProduct(circumcenter.subtractPosition(vertex2))).abs() / 2.0);
        
        double v3Area = (((v1Half.subtractPosition(vertex3)).crossProduct(circumcenter.subtractPosition(vertex3))).abs() / 2.0) +
                (((v2Half.subtractPosition(vertex3)).crossProduct(circumcenter.subtractPosition(vertex3))).abs() / 2.0);
        voronoiPoint = new Vector3d(v1Area, v2Area, v3Area);
        
        return voronoiPoint;
    }
    
    @Override
    public String toString() {
        String ret = "TRI {" + System.lineSeparator();
        for (MeshPoint p: this) {
            ret += "   " + p + System.lineSeparator();
        }
        ret += "}";
        return ret;
    }
    
    /**
     * Computes perpendicular projection from 3D point to the plane of triangle
     *
     * @param point   vertex from which the projection is created
     * @return projection to plane of triangle
     * @throws {@code NullPointerException} if the point is {@code null}
     */
    protected Vector3d getProjectionToTrianglePlane(Vector3d point) {
        Vector3d normal = computeNormal();
        Vector3d helperVector = new Vector3d(point);
        helperVector.sub(getVertex1());
        double dist = helperVector.dot(normal);

        Vector3d projection = new Vector3d(point);
        projection.scaleAdd(-dist, normal, projection);
        return projection;
    }
    
    /**
     * Checks if a point lying on the plane of the triangle lies also inside 
     * the triangle boundaries.
     *
     * @param point    Point located on the plane of the triangle
     * @return true if point is in triangle, false otherwise
     */
    protected boolean isPointInTriangle(Vector3d point) {
        List<Vector3d> pointToVertices = new ArrayList<>();
        for (MeshPoint p: this) {
            Vector3d v = new Vector3d(p.getPosition());
            v.sub(point);
            pointToVertices.add(v);
        }
        
        double angleSum = 0;
        for (int i = 0; i < 3; i++) {
            angleSum += pointToVertices.get(i).angle(pointToVertices.get((i + 1) % 3));
        }
        angleSum -= 2 * Math.PI;
        
        return -EPS < angleSum && angleSum < EPS;
    }
    
    /**
     * Computes projection of a 3D point laying on the plane of the triangle 
     * to the nearest edge of the triangle
     *
     * @param point    point laying on the plane of the triangle
     * @return perpendicular projection to the nearest edge
     */
    protected Vector3d getProjectionToClosestEdge(Vector3d point) {
        double minDistance = Double.POSITIVE_INFINITY;
        Vector3d closestProjection = null;

        Vector3d projection = getProjectionToEdge(point, getVertex2(), getVertex1());
        Vector3d aux = new Vector3d(point);
        aux.sub(projection);
        double dist = aux.length();
        if (dist < minDistance) {
            minDistance = dist;
            closestProjection = projection;
        }
        
        projection = getProjectionToEdge(point, getVertex3(), getVertex2());
        aux.sub(point, projection);
        dist = aux.length();
        if (dist < minDistance) {
            minDistance = dist;
            closestProjection = projection;
        }
        
        projection = getProjectionToEdge(point, getVertex1(), getVertex3());
        aux.sub(point, projection);
        dist = aux.length();
        if (dist < minDistance) {
            minDistance = dist;
            closestProjection = projection;
        }
        
        return closestProjection;
    }
    
    /**
     * Computes projection to edge
     *
     * @param point Point laying on the plane of the triangle
     * @param v1 first vertex of edge
     * @param v2 second vertex of edge
     * @return projection to edge
     */
    protected Vector3d getProjectionToEdge(Vector3d point, Vector3d v1, Vector3d v2) {
        Vector3d ab = new Vector3d(v2);
        ab.sub(v1);
        Vector3d ap = new Vector3d(point);
        ap.sub(v1);
        
        double abLenSquared = ab.lengthSquared();
        
        double t = ab.dot(ap) / abLenSquared;
        Vector3d projection = new Vector3d(
                v1.x + t * ab.x,
                v1.y + t * ab.y,
                v1.z + t * ab.z);

        Vector3d projectionToEdgeVertex1 = new Vector3d(v1);
        projectionToEdgeVertex1.sub(projection);
        double len1 = projectionToEdgeVertex1.length();
        
        Vector3d projectionToEdgeVertex2 = new Vector3d(v2);
        projectionToEdgeVertex2.sub(projection);
        double len2 = projectionToEdgeVertex2.length();
        
        if (Math.abs((len1 + len2) - Math.sqrt(abLenSquared)) < EPS) {
            return projection;
        }

        if (len1 < len2) {
            return v1;
        }

        return v2;
    }
    
}
