package cz.fidentis.analyst.face;

import com.google.common.eventbus.EventBus;
import cz.fidentis.analyst.mesh.core.MeshFacet;
import cz.fidentis.analyst.mesh.core.MeshModel;
import cz.fidentis.analyst.mesh.events.MeshEvent;
import cz.fidentis.analyst.mesh.events.MeshListener;
import cz.fidentis.analyst.mesh.io.MeshObjLoader;
import cz.fidentis.analyst.symmetry.Plane;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;

/**
 * This class encapsulates data for a 3D scan of a single human face.
 * <p>
 * Changes in the human face and its data structures (e.g., mesh model, etc.)
 * can be monitored by listeners. Listeners have to implement the 
 * {@link cz.fidentis.analyst.face.HumanFaceListener} interface and they have to be
 * registered using the {@link cz.fidentis.analyst.face.HumanFace#registerListener} method.
 * Then they are informed about changes in the human automatically via methods 
 * prescribed by the interface.
 * </p>
 * <p>
 * Events fired by the class:
 * <ul>
 * <li>{@link cz.fidentis.analyst.mesh.events.MeshEvent} or its sub-type if 
 * the mesh model changes (see {@link cz.fidentis.analyst.mesh.core.MeshModel}
 * documentation for monitored changes).</li>
 * </ul>
 * </p>
 * 
 * @author Radek Oslejsek
 */
public class HumanFace implements MeshListener {
    
    private MeshModel meshModel;

    private Plane symmetryPlane;

    private MeshFacet cuttingPlane;
    
    private EventBus eventBus = new EventBus();
    
    /**
     * Reads a 3D human phase from file.
     * 
     * @param file OBJ file
     * @throws IOException on I/O failure
     */
    public HumanFace(File file) throws IOException {
        this(new FileInputStream(file));
    }
    
    /**
     * Reads a 3D human phase from file.
     * 
     * @param is input stream with OBJ data
     * @throws IOException on I/O failure
     */
    public HumanFace(InputStream is) throws IOException {
        meshModel = MeshObjLoader.read(is);
        meshModel.simplifyModel();
        meshModel.registerListener(this);
    }
    
    /**
     * Returns the triangular mesh model of the human face.
     * 
     * @return the triangular mesh model of the human face
     */
    public MeshModel getMeshModel() {
        return meshModel;
    }

    /**
     * Registers listeners (objects concerned in the human face changes) to receive events.
     * If listener is {@code null}, no exception is thrown and no action is taken.
     * 
     * @param listener Listener concerned in the human face changes.
     */
    public void registerListener(MeshListener listener) {
        eventBus.register(listener);
    }
    
    /**
     * Unregisters listeners from receiving events.
     * 
     * @param listener Registered listener
     */
    public void unregisterListener(MeshListener listener) {
        eventBus.unregister(listener);
    }

    /**
     * Captures events fired by {@link cz.fidentis.analyst.mesh.core.MeshModel} and
     * redirects them to our listeners.
     * 
     * @param event A fired event.
     */
    @Override
    public void meshEvent(MeshEvent event) {
        eventBus.post(event);
    }

    /**
     *
     * @param plane The new symmetry plane
     */
    public void setSymmetryPlane(Plane plane) {
        symmetryPlane = plane;
    }

    /**
     *
     * @return The face's symmetry plane
     */
    public Plane getSymmetryPlane() {
        return symmetryPlane;
    }

    /**
     *
     * @param facet MeshFacet of the new cutting plane
     */
    public void setCuttingPlane(MeshFacet facet) {
        cuttingPlane = facet;
    }

    /**
     *
     * @return The face's cutting plane
     */
    public MeshFacet getCuttingPlane() {
        return cuttingPlane;
    }
}
