package cz.fidentis.analyst.mesh.core;

import javax.vecmath.Vector3d;

/**
 * MeshPoint represents a point with position, normal, and texture coordinates
 *
 * @author Matej Lukes
 */
public interface MeshPoint {
    
    /**
     * @return normal
     */
    Vector3d getNormal();

    /**
     * @return position
     */
    Vector3d getPosition();

    /**
     * @return texture coordinates
     */
    Vector3d getTexCoord();

    /**
     * returns new instance of MeshPoint with subtracted position
     *
     * @param subtrahend position to be subtracted
     * @return subtracted MeshPoint
     */
    MeshPoint subtractPosition(MeshPoint subtrahend);

    /**
     * returns new instance of MeshPoint with subtracted position
     *
     * @param subtrahend position to be subtracted
     * @return subtracted MeshPoint
     */
    MeshPoint subtractPosition(Vector3d subtrahend);

    /**
     * returns new instance of MeshPoint with added position
     *
     * @param addend position to be added
     * @return added MeshPoint
     */
    MeshPoint addPosition(MeshPoint addend);

    /**
     * returns new instance of MeshPoint with added position
     *
     * @param addend position to be added
     * @return added MeshPoint
     */
    MeshPoint addPosition(Vector3d addend);

    /**
     * returns new instance of MeshPoint with multiplied position by number
     * 
     * @param number Number for multiplying
     * @return multiplied MeshPoint
     */
    MeshPoint multiplyPosition(double number);

    /**
     * returns new instance of MeshPoint with divided position by number
     * 
     * @param number Number for division
     * @return divided MeshPoint
     */
    MeshPoint dividePosition(double number);
    
    /**
     * Returns the cross product of two points.
     * 
     * @param meshPoint Second argument of the cross product operation.
     * @return MeshPoint representing the resulting vector.
     */
    MeshPoint crossProduct(MeshPoint meshPoint);

    /**
     * returns the dot product of two points
     * 
     * @param meshPoint Second argument of the dot product operation
     * @return dot product of two instances of MeshPoint 
     */
    double dotProduct(MeshPoint meshPoint);
    
    /**
     * returns absolute value of MeshPoint
     * 
     * @return absolute value of MeshPoint
     */
    double abs();

    /**
     * returns new instance of MeshPoint with subtracted normal
     *
     * @param subtrahend normal to be subtracted
     * @return subtracted MeshPoint
     */
    MeshPoint subtractNormal(MeshPoint subtrahend);

    /**
     * returns new instance of MeshPoint with subtracted normal
     *
     * @param subtrahend normal to be subtracted
     * @return subtracted MeshPoint
     */
    MeshPoint subtractNormal(Vector3d subtrahend);

    /**
     * returns new instance of MeshPoint with added normal
     *
     * @param addend normal to be added
     * @return added MeshPoint
     */
    MeshPoint addNormal(MeshPoint addend);

    /**
     * returns new instance of MeshPoint with added normal
     *
     * @param addend normal to be added
     * @return added MeshPoint
     */
    MeshPoint addNormal(Vector3d addend);

}
