package cz.fidentis.analyst.mesh.core;

import java.util.List;
import javax.vecmath.Vector3d;

/**
 * @author Natalia Bebjakova
 * 
 * 3D bounding box (cube) of <code>MeshPoint</code>s.
 */

public class BoundingBox {
    
    private final MeshPoint maxPoint;
    private final MeshPoint minPoint;
    private final MeshPoint midPoint;
    private final double maxDiag;
    
    /** 
     * Creates bounding box from given mesh points.
     * @param points List of mesh points, must not be null or empty
     * @throws IllegalArgumentException if the @code{points} param is null or empty
     */
    public BoundingBox(List<MeshPoint> points) {
        if (points == null || points.isEmpty()) {
            throw new IllegalArgumentException("points");
        }
        
        minPoint = new MeshPointImpl(new Vector3d(Double.POSITIVE_INFINITY,Double.POSITIVE_INFINITY,Double.POSITIVE_INFINITY), null, null);
        maxPoint = new MeshPointImpl(new Vector3d(Double.NEGATIVE_INFINITY,Double.NEGATIVE_INFINITY,Double.NEGATIVE_INFINITY), null, null);
 
        for (int i = 0; i < points.size(); i++) {
            MeshPoint point = points.get(i);
            
            minPoint.getPosition().x = Math.min(minPoint.getPosition().x, point.getPosition().x);
            minPoint.getPosition().y = Math.min(minPoint.getPosition().y, point.getPosition().y);
            minPoint.getPosition().z = Math.min(minPoint.getPosition().z, point.getPosition().z);
            
            maxPoint.getPosition().x = Math.max(maxPoint.getPosition().x, point.getPosition().x);
            maxPoint.getPosition().y = Math.max(maxPoint.getPosition().y, point.getPosition().y);
            maxPoint.getPosition().z = Math.max(maxPoint.getPosition().z, point.getPosition().z);                    
        }
        
        midPoint = (minPoint.addPosition(maxPoint)).multiplyPosition(0.5);
        MeshPoint diag = maxPoint.subtractPosition(minPoint);
        this.maxDiag = diag.abs();
    }        

    /**
     * Return the upper-bound corner of the bounding cube
     * @return max point of the bounding box
     */
    public MeshPoint getMaxPoint() {
        return maxPoint;
    }
    
    /**
     * Return centroid of the bounding cube.
     * @return middle point of the bounding box
     */
    public MeshPoint getMidPoint() {
        return midPoint;
    }
    
    /**
     * Return the lower-bound corner of the bounding cube
     * @return min point of the bounding box
     */
    public MeshPoint getMinPoint() {
        return minPoint;
    }
    
    /**
     * Return volume diagonal of the bounding box.
     * @return maximal diagonal of bounding box
     */
    public double getMaxDiag() {
        return maxDiag;
    }
    
    /**
     * Returns description of BoundignBox.
     * 
     * @return String representation of the bounding box
     */
    @Override
    public String toString() {
        String str = "BoundingBox: ";
        str += System.lineSeparator();
        str += "\t" + "- min point : " + this.minPoint + System.lineSeparator();
        str += "\t" + "- max point : " + this.maxPoint + System.lineSeparator();
        str += "\t" + "- mid point : " + this.midPoint + System.lineSeparator();
        str += "\t" + "- max diag  : " + this.maxDiag + System.lineSeparator();
        return str;
    }      
}  