package cz.fidentis.analyst.gui.scene;

import com.jogamp.opengl.GL2;
import cz.fidentis.analyst.feature.FeaturePoint;
import cz.fidentis.analyst.mesh.core.MeshFacet;
import cz.fidentis.analyst.mesh.core.MeshModel;
import java.awt.Color;
import java.util.ArrayList;
import java.util.List;
import javax.vecmath.Vector3d;

/**
 * A drawable triangular mesh, i.e., a mesh model with drawing information like 
 * material, transparency, color, relative transformations in the scene etc. 
 * This class encapsulates rendering state and parameters,
 * 
 * @author Radek Oslejsek
 * @author Richard Pajersky
 */
public class DrawableMesh {
    
    private final MeshModel model;
    
    private boolean display = true;
    
    /* material info */
    /**
     * {@link Color} of mesh
     */
    private Color color = new Color(255, 255, 255);
    /**
     * {@link Color} of highlights
     */
    private Color highlights = new Color(0, 0, 0, 1);
    /**
     * Transparency of mesh
     */
    private float transparency = 1;
    
    /* transformation info */
    /**
     * Translation
     */
    private Vector3d translation = new Vector3d(0, 0, 0);
    /**
     * Rotation
     */
    private Vector3d rotation = new Vector3d(0, 0, 0);
    /**
     * Scale
     */
    private Vector3d scale = new Vector3d(0, 0, 0);
    
    /* render mode */
    /**
     * Render mode to use, one from {@code GL_FILL}, {@code GL_LINE}, {@code GL_POINT}
     */
    private int renderMode = GL2.GL_FILL;
    /**
     * Flag if back should be rendered
     */
    private boolean showBackface = true;
    
    /* feature points */
    /**
     * {@link List} of feature points
     */
    private List<FeaturePoint> featurePoints = new ArrayList<>();
    /**
     * {@link List} of feature points color
     */
    private List<Color> featurePointsColor = new ArrayList<>();
    /**
     * Flag if feature points should be rendered
     */
    private boolean renderFeaturePoints = false;
    
    /**
     * Constructor. 
     * 
     * @param model Drawable mesh model
     * @throws IllegalArgumentException if the model is {@code null}
     */
    public DrawableMesh(MeshModel model) {
        if (model == null) {
            throw new IllegalArgumentException("model is null");
        }
        this.model = model;
    }
    
    /**
     * Returns list of individual facets.
     * 
     * @return list of individual facets.
     */
    public List<MeshFacet> getFacets() {
        return model.getFacets();
    }
    
    /**
     * This drawable mesh is included in the rendered scene.
     */
    public void show() {
        display = true;
    }
    
    /**
     * This drawable mesh is excluded from the rendered scene (skipped).
     */
    public void hide() {
        display = false;
    }
    
    /**
     * 
     * @return {@code true} if the object is included (rendered) in the scene.
     */
    public boolean isShown() {
        return display;
    }
    
    /**
     * Sets color
     * @param color Color
     */
    public void setColor(Color color) {
        this.color = color;
    }

    /**
     * @return {@link Color}
     */
    public Color getColor() {
        return color;
    }
    
    /**
     * @return Current value of transparency
     */
    public float getTransparency() {
        return transparency;
    }

    /**
     * Sets transparency
     * @param transparency Transparency
     */
    public void setTransparency(float transparency) {
        this.transparency = transparency;
    }
    
    /**
     * @return Current translation
     */
    public Vector3d getTranslation() {
        return translation;
    }

    /**
     * Sets tranlation
     * @param translation Translation
     */
    public void setTranslation(Vector3d translation) {
        this.translation = translation;
    }

    /**
     * @return Current rotation
     */
    public Vector3d getRotation() {
        return rotation;
    }

    /**
     * Sets rotation
     * @param rotation 
     */
    public void setRotation(Vector3d rotation) {
        this.rotation = rotation;
    }

    /**
     * @return Current scale
     */
    public Vector3d getScale() {
        return scale;
    }

    /**
     * Sets scale
     * @param scale Scale
     */
    public void setScale(Vector3d scale) {
        this.scale = scale;
    }
    
    /**
     * @return {@link MeshModel}
     */
    public MeshModel getModel() {
        return this.model;
    }

    /**
     * @return {@link Color} of highlights
     */
    public Color getHighlights() {
        return highlights;
    }

    /**
     * Sets {@link Color} of highlights
     * @param highlights 
     */
    public void setHighlights(Color highlights) {
        this.highlights = highlights;
    }
    
    /**
     * @return Value of {@link #renderMode}
     */
    public int getRenderMode() {
        return renderMode;
    }

    /**
     * Sets render mode
     * @param renderMode Render mode
     * @throws IllegalArgumentException if renderMode isn't {@code GL_FILL}, {@code GL_LINE} or {@coed GL_POINT}
     */
    public void setRenderMode(int renderMode) {
        if (renderMode != GL2.GL_FILL && 
                renderMode != GL2.GL_LINE &&
                renderMode != GL2.GL_POINT) {
            throw new IllegalArgumentException("invalid mode");
        }
        this.renderMode = renderMode;
    }

    /**
     * @return {@link List} of {@link FeaturePoint}
     */
    public List<FeaturePoint> getFeaturePoints() {
        return featurePoints;
    }

    /**
     * Sets feature points
     * @param featurePoints Feature points
     */
    public void setFeaturePoints(List<FeaturePoint> featurePoints) {
        this.featurePoints = featurePoints;
        List<Color> colors = new ArrayList<>();
        featurePoints.forEach((_item) -> {
            colors.add(this.color);
        });
        this.setFeaturePointsColor(colors);
    }   
    
    /**
     * @return {@link List} of feature points {@link Color}
     */
    public List<Color> getFeaturePointsColor() {
        return featurePointsColor;
    }

    /**
     * Sets colors of feature points
     * @param featurePointsColor Colors of feature points
     */
    public void setFeaturePointsColor(List<Color> featurePointsColor) {
        this.featurePointsColor = featurePointsColor;
    } 
    
    /**
     * @return {@code true} if feature points should be rendered
     */
    public boolean isRenderFeaturePoints() {
        return renderFeaturePoints;
    }

    /**
     * Sets if feature points should be renderred or not
     * @param renderFeaturePoints 
     */
    public void setRenderFeaturePoints(boolean renderFeaturePoints) {
        this.renderFeaturePoints = renderFeaturePoints;
    }

    /**
     * @return {@code true} if back face shoud be shown
     */
    public boolean isShowBackface() {
        return showBackface;
    }

    /**
     * Sets if back face should be shown or not
     * @param showBackface 
     */
    public void setShowBackface(boolean showBackface) {
        this.showBackface = showBackface;
    }
    
}
