package cz.fidentis.analyst.distance;

import cz.fidentis.analyst.canvas.Canvas;
import cz.fidentis.analyst.core.LoadedActionEvent;
import cz.fidentis.analyst.core.ControlPanelAction;
import cz.fidentis.analyst.feature.FeaturePoint;
import cz.fidentis.analyst.feature.FeaturePointType;
import cz.fidentis.analyst.mesh.core.MeshFacet;
import cz.fidentis.analyst.scene.DrawableFeaturePoints;
import cz.fidentis.analyst.visitors.face.HausdorffDistancePrioritized;
import cz.fidentis.analyst.visitors.mesh.HausdorffDistance.Strategy;
import java.awt.Color;
import java.awt.event.ActionEvent;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import javax.swing.JComboBox;
import javax.swing.JTabbedPane;
import javax.swing.JTextField;
import javax.swing.JToggleButton;

/**
 * Action listener for the curvature computation.
 * 
 * @author Radek Oslejsek
 * @author Daniel Schramm
 */
public class DistanceAction extends ControlPanelAction {
    
    /*
     * Attributes handling the state
     */
    private HausdorffDistancePrioritized visitor = null;
    private final Map<FeaturePointType, Double> featurePointTypes = new HashMap<>();
    private String strategy = DistancePanel.STRATEGY_POINT_TO_POINT;
    private boolean relativeDist = false;
    private boolean weightedDist = false;
    private boolean heatmapRender = false;
    
    private Map<MeshFacet, List<Double>> weightedHausdorffDistance = null;
    
    private final DistancePanel controlPanel;
    
    private static final Color FEATURE_POINT_HIGHLIGHT_COLOR = Color.MAGENTA;
    
    /**
     * Constructor.
     * 
     * @param canvas OpenGL canvas
     * @param topControlPanel Top component for placing control panels
     */
    public DistanceAction(Canvas canvas, JTabbedPane topControlPanel) {
        super(canvas, topControlPanel);
        this.controlPanel = new DistancePanel(this, getSecondaryFeaturePoints().getFeaturePoints());
        
        // Place control panel to the topControlPanel
        topControlPanel.addTab(controlPanel.getName(), controlPanel.getIcon(), controlPanel);
        topControlPanel.addChangeListener(e -> {
            // If the distance panel is focused...
            if (((JTabbedPane) e.getSource()).getSelectedComponent() instanceof DistancePanel) {
                // ... display heatmap and feature points relevant to the Hausdorff distance
                getCanvas().getScene().setDefaultColors();
                final List<FeaturePoint> secondaryFeaturePoints = getSecondaryFeaturePoints().getFeaturePoints();
                for (int i = 0; i < secondaryFeaturePoints.size(); i++) {
                    if (featurePointTypes.containsKey(secondaryFeaturePoints.get(i).getFeaturePointType())) {
                        getSecondaryFeaturePoints().setColor(i, FEATURE_POINT_HIGHLIGHT_COLOR);
                    }
                }
                visitor = null; // recompute (position of faces could have been changed in registration)
                calculateHausdorffDistance();
                getSecondaryDrawableFace().setRenderHeatmap(heatmapRender);
                renderScene();
            }
        });
        topControlPanel.setSelectedComponent(controlPanel); // Focus Hausdorff distance panel
    }

    @Override
    public void actionPerformed(ActionEvent ae) {
        final String action = ae.getActionCommand();
        
        switch (action) {
            case DistancePanel.ACTION_COMMAND_SHOW_HIDE_HEATMAP:
                if (((JToggleButton) ae.getSource()).isSelected()) {
                    calculateHausdorffDistance();
                    heatmapRender = true;
                } else {
                    heatmapRender = false;
                }
                getSecondaryDrawableFace().setRenderHeatmap(heatmapRender);
                break;
            case DistancePanel.ACTION_COMMAND_SET_DISTANCE_STRATEGY:
                strategy = (String) ((JComboBox) ae.getSource()).getSelectedItem();
                this.visitor = null; // recompute
                calculateHausdorffDistance();
                break;
            case DistancePanel.ACTION_COMMAND_RELATIVE_ABSOLUTE_DIST:
                this.relativeDist = ((JToggleButton) ae.getSource()).isSelected();
                this.visitor = null; // recompute
                calculateHausdorffDistance();
                break;
            case DistancePanel.ACTION_COMMAND_FEATURE_POINT_HIGHLIGHT:
                highlightFeaturePoint((LoadedActionEvent) ae);
                break;
            case DistancePanel.ACTION_COMMAND_FEATURE_POINT_RESIZE:
                resizeFeaturePoint((LoadedActionEvent) ae);
                break;
            case DistancePanel.ACTION_COMMAND_WEIGHTED_DISTANCE:
                this.weightedDist = ((JToggleButton) ae.getSource()).isSelected();
                calculateHausdorffDistance();
                break;
            case DistancePanel.ACTION_COMMAND_DISTANCE_RECOMPUTE:
                this.visitor = null; // recompute
                calculateHausdorffDistance();
                break;
            default:
                // to nothing
        }
        renderScene();
    }
    
    /**
     * (Re)calculates the Hausdorff distance and updates the heat map of the secondary face
     * as well as values of all appropriate GUI elements of {@link DistancePanel}.
     */
    private void calculateHausdorffDistance() {
        final Strategy useStrategy;
        switch (strategy) {
            case DistancePanel.STRATEGY_POINT_TO_POINT:
                useStrategy = Strategy.POINT_TO_POINT;
                break;
            case DistancePanel.STRATEGY_POINT_TO_TRIANGLE:
                useStrategy = Strategy.POINT_TO_TRIANGLE_APPROXIMATE;
                break;
            default:
                throw new UnsupportedOperationException(strategy);
        }
        
        if (visitor == null) {
            this.visitor = new HausdorffDistancePrioritized(getPrimaryDrawableFace().getModel(),
                    featurePointTypes,
                    useStrategy,
                    relativeDist,
                    true);
            getSecondaryDrawableFace().getHumanFace().accept(visitor);
            
            weightedHausdorffDistance = getWeightedDistance();
            
            // Update GUI elements that display the calculated Hausdorff distance metrics
            setFeaturePointWeigths();
            setHausdorffDistanceStatistics();
        }
        
        getSecondaryDrawableFace().setHeatMap(weightedDist ? weightedHausdorffDistance : visitor.getDistances());
    }
    
    /**
     * Calculates weighted Hausdorff distance of the face.
     * 
     * @return weighted Hausdorff distance
     */
    private Map<MeshFacet, List<Double>> getWeightedDistance() {
        final Map<MeshFacet, List<Double>> weightedDistances = new HashMap<>(visitor.getDistances());
        final Map<MeshFacet, List<Double>> mergedPriorities = visitor.getMergedPriorities()
                .get(getSecondaryDrawableFace().getHumanFace());

        // Merge the map of distances with the map of priorities
        for (final Map.Entry<MeshFacet, List<Double>> facetPriorities: mergedPriorities.entrySet()) {
            weightedDistances.merge(
                    facetPriorities.getKey(),
                    facetPriorities.getValue(),
                    (distancesList, prioritiesList) ->
                            IntStream.range(0, distancesList.size())
                                    .mapToDouble(i -> distancesList.get(i) * prioritiesList.get(i))
                                    .boxed()
                                    .collect(Collectors.toList()));
        }

        return weightedDistances;
    }
    
    /**
     * Updates the GUI elements of {@link DistancePanel} that display
     * the weights of feature points used to calculate the weighted Hausdorff distance.
     */
    private void setFeaturePointWeigths() {
        controlPanel.updateFeaturePointWeights(
                visitor.getFeaturePointWeights()
                        .get(getSecondaryDrawableFace().getHumanFace()) // Get FP weights for the secondary face
                        .entrySet()
                        .stream()
                        .collect(
                                Collectors.toMap(
                                        Map.Entry::getKey,            // For each FP type at the secondary face...
                                        weights -> weights.getValue() // ... compute average FP weight over all its facets
                                                .values()
                                                .stream()
                                                .mapToDouble(weight -> weight)
                                                .average()
                                                .orElse(Double.NaN))));
    }

    /**
     * Updates the GUI elements of {@link DistancePanel} elements that display
     * statistical data about the calculated Hausdorff distance.
     */
    private void setHausdorffDistanceStatistics() {
        controlPanel.updateHausdorffDistanceStats(
                visitor.getDistances()
                        .values()
                        .stream()
                        .flatMap(List::stream)
                        .mapToDouble(distance -> distance)
                        .summaryStatistics(),
                weightedHausdorffDistance
                        .values()
                        .stream()
                        .flatMap(List::stream)
                        .mapToDouble(distance -> distance)
                        .summaryStatistics()
        );
    }
    
    /**
     * Changes the colour of the secondary face's feature point at the given index.
     * The index is received as the data payload of {@code actionEvent}.
     * 
     * @param actionEvent Action event with the index of the feature point as its payload data
     */
    private void highlightFeaturePoint(LoadedActionEvent actionEvent) {
        final int index = (int) actionEvent.getData();
        final FeaturePointType fpType = getTypeOfFeaturePoint(index);
        final DrawableFeaturePoints secondaryFeaturePoints = getSecondaryFeaturePoints();
        
        if (((JToggleButton) actionEvent.getSource()).isSelected()) {
            secondaryFeaturePoints.setColor(index, FEATURE_POINT_HIGHLIGHT_COLOR);
            featurePointTypes.put(fpType, secondaryFeaturePoints.getSize(index));
        } else {
            secondaryFeaturePoints.resetColorToDefault(index);
            featurePointTypes.remove(fpType);
        }
    }

    /**
     * Changes the size of the secondary face's feature point at the given index.
     * The index is received as the data payload of {@code actionEvent}.
     * 
     * @param actionEvent Action event with the index of the feature point as its payload data
     */
    private void resizeFeaturePoint(LoadedActionEvent actionEvent) {
        final int index = (int) actionEvent.getData();
        final double size = Double.parseDouble(((JTextField) actionEvent.getSource()).getText());
        
        getSecondaryFeaturePoints().setSize(index, size);
        featurePointTypes.replace(getTypeOfFeaturePoint(index), size);
    }
    
    /**
     * Returns type of the feature point at the given index in the secondary face.
     * 
     * @param index Index of the feature point
     * @return Type of the feature point or {@code null}
     */
    private FeaturePointType getTypeOfFeaturePoint(int index) {
        final List<FeaturePoint> featurePoints = getSecondaryFeaturePoints().getFeaturePoints();
        if (index < 0 || index >= featurePoints.size()) {
            return null;
        }
        
        return featurePoints.get(index)
                .getFeaturePointType();
    }
}
