package cz.fidentis.analyst.core;

import cz.fidentis.analyst.Logger;
import org.netbeans.api.settings.ConvertAsProperties;
import org.openide.awt.ActionID;
import org.openide.awt.ActionReference;
import org.openide.windows.TopComponent;
import org.openide.util.NbBundle.Messages;
import cz.fidentis.analyst.Project;
import cz.fidentis.analyst.face.HumanFace;
import cz.fidentis.analyst.dashboard.ModelsTableModel;
import cz.fidentis.analyst.dashboard.FaceStatePanel;
import cz.fidentis.analyst.dashboard.FilterPanel;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.swing.AbstractAction;
import javax.swing.JOptionPane;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.filechooser.FileNameExtensionFilter;
import org.openide.filesystems.FileChooserBuilder;

/**
 * The main panel enabling analysts to select the primary and secondary faces,
 * and to perform basic batch processing. This panel also serves as an entry
 * point for detailed face analysis and face-to-face comparison.
 *
 * @author Matej Kovar
 */
@ConvertAsProperties(
        dtd = "-//cz.fidentis.analyst.gui//Dashboard//EN",
        autostore = false
)
@TopComponent.Description(
        preferredID = "ProjectTopComp",
        //iconBase="SET/PATH/TO/ICON/HERE",
        persistenceType = TopComponent.PERSISTENCE_ALWAYS
)
@TopComponent.Registration(mode = "editor", openAtStartup = true)
@ActionID(category = "Window", id = "cz.fidentis.analyst.gui.ProjectTopComp")
@ActionReference(path = "Menu/Window" /*, position = 333 */)
@TopComponent.OpenActionRegistration(
        displayName = "#CTL_ProjectTopCompAction",
        preferredID = "ProjectTopComp"
)
@Messages({
    "CTL_ProjectTopCompAction=Project",
    "CTL_ProjectTopCompTopComponent=Project",
    "HINT_ProjectTopCompTopComponent=This is a Project window"
})
public final class ProjectTopComp extends TopComponent {

    private final Project project;
    
    private Map<HumanFace, SingleFaceTab> singleFaceTabs = new HashMap<>();
    private Map<HumanFace, FaceToFaceTab> faceToFaceTabs = new HashMap<>();
    
    private ModelsTableModel model = new ModelsTableModel(new Object[]{"", "Models", "KD-tree"}, 0);
    
    private FilterPanel fp;
    
    /* List of indexes of selected Rows */
    private List<Integer> selectedRows = new ArrayList<>();
    
    /**
     * Creates new ProjectTopComp, initializes new project
     */
    public ProjectTopComp() {
        project = new Project();
        initComponents();
        setName(Bundle.CTL_ProjectTopCompTopComponent());
        setToolTipText(Bundle.HINT_ProjectTopCompTopComponent());
        putClientProperty(TopComponent.PROP_CLOSING_DISABLED, Boolean.TRUE);
        putClientProperty(TopComponent.PROP_DRAGGING_DISABLED, Boolean.TRUE);
        putClientProperty(TopComponent.PROP_UNDOCKING_DISABLED, Boolean.TRUE);
        
        ActionListener listener = new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                applyFilter();
            }
        };
        
        fp = new FilterPanel(filterPanel, listener);
                
        // Execute infinite OutputWindowThread that redirects messages logged
        // via Logger into the standard output window
        OutputWindowThread.execute();

    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        mainScrollPanel = new javax.swing.JScrollPane();
        mainPanel = new javax.swing.JPanel();
        buttonsPanel = new javax.swing.JPanel();
        addButton1 = new javax.swing.JButton();
        removeButton1 = new javax.swing.JButton();
        selectAllButton1 = new javax.swing.JButton();
        deselectAllButton1 = new javax.swing.JButton();
        inflateButton1 = new javax.swing.JButton();
        oneOnOneButton1 = new javax.swing.JButton();
        analyseButton1 = new javax.swing.JButton();
        manyToManyButton = new javax.swing.JButton();
        faceTableScrollPanel = new javax.swing.JScrollPane();
        table = new javax.swing.JTable();
        filterPanel = new javax.swing.JPanel();
        infoPanel = new javax.swing.JPanel();

        setOpaque(true);

        buttonsPanel.setBorder(javax.swing.BorderFactory.createEtchedBorder());
        buttonsPanel.setMinimumSize(new java.awt.Dimension(0, 0));
        buttonsPanel.setLayout(new java.awt.GridBagLayout());

        addButton1.setFont(new java.awt.Font("Tahoma", 0, 12)); // NOI18N
        org.openide.awt.Mnemonics.setLocalizedText(addButton1, org.openide.util.NbBundle.getMessage(ProjectTopComp.class, "ProjectTopComp.addButton1.text")); // NOI18N
        addButton1.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                addButton1MouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.ipadx = 20;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(16, 16, 13, 4);
        buttonsPanel.add(addButton1, gridBagConstraints);

        removeButton1.setFont(new java.awt.Font("Tahoma", 0, 12)); // NOI18N
        org.openide.awt.Mnemonics.setLocalizedText(removeButton1, org.openide.util.NbBundle.getMessage(ProjectTopComp.class, "ProjectTopComp.removeButton1.text")); // NOI18N
        removeButton1.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                removeButton1MouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(16, 22, 13, 4);
        buttonsPanel.add(removeButton1, gridBagConstraints);

        selectAllButton1.setFont(new java.awt.Font("Tahoma", 0, 12)); // NOI18N
        org.openide.awt.Mnemonics.setLocalizedText(selectAllButton1, org.openide.util.NbBundle.getMessage(ProjectTopComp.class, "ProjectTopComp.selectAllButton1.text")); // NOI18N
        selectAllButton1.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                selectAllButton1MouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(16, 22, 13, 4);
        buttonsPanel.add(selectAllButton1, gridBagConstraints);

        deselectAllButton1.setFont(new java.awt.Font("Tahoma", 0, 12)); // NOI18N
        org.openide.awt.Mnemonics.setLocalizedText(deselectAllButton1, org.openide.util.NbBundle.getMessage(ProjectTopComp.class, "ProjectTopComp.deselectAllButton1.text")); // NOI18N
        deselectAllButton1.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                deselectAllButton1MouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(16, 22, 13, 4);
        buttonsPanel.add(deselectAllButton1, gridBagConstraints);

        inflateButton1.setFont(new java.awt.Font("Tahoma", 0, 12)); // NOI18N
        org.openide.awt.Mnemonics.setLocalizedText(inflateButton1, org.openide.util.NbBundle.getMessage(ProjectTopComp.class, "ProjectTopComp.inflateButton1.text")); // NOI18N
        inflateButton1.setAlignmentX(0.5F);
        inflateButton1.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                inflateButton1MouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 4;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(16, 22, 13, 4);
        buttonsPanel.add(inflateButton1, gridBagConstraints);

        oneOnOneButton1.setFont(new java.awt.Font("Tahoma", 1, 12)); // NOI18N
        org.openide.awt.Mnemonics.setLocalizedText(oneOnOneButton1, org.openide.util.NbBundle.getMessage(ProjectTopComp.class, "ProjectTopComp.oneOnOneButton1.text")); // NOI18N
        oneOnOneButton1.setAlignmentX(0.5F);
        oneOnOneButton1.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                oneOnOneButton1MouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 5;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(16, 55, 13, 4);
        buttonsPanel.add(oneOnOneButton1, gridBagConstraints);

        analyseButton1.setFont(new java.awt.Font("Tahoma", 1, 12)); // NOI18N
        org.openide.awt.Mnemonics.setLocalizedText(analyseButton1, org.openide.util.NbBundle.getMessage(ProjectTopComp.class, "ProjectTopComp.analyseButton1.text")); // NOI18N
        analyseButton1.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                analyseButton1MouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 6;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.insets = new java.awt.Insets(16, 16, 13, 4);
        buttonsPanel.add(analyseButton1, gridBagConstraints);

        manyToManyButton.setFont(new java.awt.Font("Ubuntu", 1, 14)); // NOI18N
        org.openide.awt.Mnemonics.setLocalizedText(manyToManyButton, org.openide.util.NbBundle.getMessage(ProjectTopComp.class, "ProjectTopComp.manyToManyButton.text")); // NOI18N
        manyToManyButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                manyToManyButtonMouseClicked(evt);
            }
        });
        buttonsPanel.add(manyToManyButton, new java.awt.GridBagConstraints());

        faceTableScrollPanel.setPreferredSize(new java.awt.Dimension(812, 750));

        table.setSize(faceTableScrollPanel.getWidth(), faceTableScrollPanel.getHeight());
        table.setFont(new java.awt.Font("Tahoma", 0, 18)); // NOI18N
        table.getTableHeader().setOpaque(false);
        table.getTableHeader().setBackground(new java.awt.Color(204,204,204));
        table.getTableHeader().setFont(new java.awt.Font("Tahoma", 0, 18));
        model.addTableModelListener(new TableModelListener() {

            public void tableChanged(TableModelEvent e) {
                jTable1TableChanged(e);
            }
        });
        table.setModel(model);
        table.getColumnModel().getColumn(0).setMaxWidth(50);
        table.getColumnModel().getColumn(2).setMaxWidth(75);
        table.getTableHeader().getColumnModel().getColumn(0).setMaxWidth(50);
        table.getTableHeader().getColumnModel().getColumn(2).setMaxWidth(75);
        table.setDragEnabled(true);
        table.setRowHeight(40);
        table.setSelectionBackground(new java.awt.Color(102, 204, 255));
        table.setSelectionMode(javax.swing.ListSelectionModel.SINGLE_SELECTION);
        table.getTableHeader().setReorderingAllowed(false);
        table.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                tableMouseClicked(evt);
            }
        });
        faceTableScrollPanel.setViewportView(table);

        filterPanel.setBorder(javax.swing.BorderFactory.createEtchedBorder());
        filterPanel.setMaximumSize(new java.awt.Dimension(342, 424));
        filterPanel.setPreferredSize(new java.awt.Dimension(342, 424));

        javax.swing.GroupLayout filterPanelLayout = new javax.swing.GroupLayout(filterPanel);
        filterPanel.setLayout(filterPanelLayout);
        filterPanelLayout.setHorizontalGroup(
            filterPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGap(0, 359, Short.MAX_VALUE)
        );
        filterPanelLayout.setVerticalGroup(
            filterPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGap(0, 420, Short.MAX_VALUE)
        );

        infoPanel.setBorder(javax.swing.BorderFactory.createEtchedBorder());
        infoPanel.setPreferredSize(new java.awt.Dimension(349, 363));

        javax.swing.GroupLayout infoPanelLayout = new javax.swing.GroupLayout(infoPanel);
        infoPanel.setLayout(infoPanelLayout);
        infoPanelLayout.setHorizontalGroup(
            infoPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGap(0, 0, Short.MAX_VALUE)
        );
        infoPanelLayout.setVerticalGroup(
            infoPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGap(0, 312, Short.MAX_VALUE)
        );

        infoPanel.setVisible(false);

        javax.swing.GroupLayout mainPanelLayout = new javax.swing.GroupLayout(mainPanel);
        mainPanel.setLayout(mainPanelLayout);
        mainPanelLayout.setHorizontalGroup(
            mainPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(mainPanelLayout.createSequentialGroup()
                .addContainerGap()
                .addGroup(mainPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING, false)
                    .addComponent(faceTableScrollPanel, javax.swing.GroupLayout.PREFERRED_SIZE, 0, Short.MAX_VALUE)
                    .addComponent(buttonsPanel, javax.swing.GroupLayout.DEFAULT_SIZE, 792, Short.MAX_VALUE))
                .addGap(18, 18, Short.MAX_VALUE)
                .addGroup(mainPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING, false)
                    .addComponent(filterPanel, javax.swing.GroupLayout.DEFAULT_SIZE, 363, Short.MAX_VALUE)
                    .addComponent(infoPanel, javax.swing.GroupLayout.DEFAULT_SIZE, 363, Short.MAX_VALUE))
                .addGap(20, 20, 20))
        );
        mainPanelLayout.setVerticalGroup(
            mainPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(mainPanelLayout.createSequentialGroup()
                .addContainerGap()
                .addComponent(buttonsPanel, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(mainPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING, false)
                    .addGroup(mainPanelLayout.createSequentialGroup()
                        .addComponent(filterPanel, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addGap(28, 28, 28)
                        .addComponent(infoPanel, javax.swing.GroupLayout.PREFERRED_SIZE, 316, javax.swing.GroupLayout.PREFERRED_SIZE))
                    .addComponent(faceTableScrollPanel, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
                .addContainerGap())
        );

        mainScrollPanel.setViewportView(mainPanel);

        mainScrollPanel.setSize(ControlPanel.CONTROL_PANEL_WIDTH, ControlPanel.HEIGHT);

        javax.swing.GroupLayout layout = new javax.swing.GroupLayout(this);
        this.setLayout(layout);
        layout.setHorizontalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addComponent(mainScrollPanel, javax.swing.GroupLayout.Alignment.TRAILING, javax.swing.GroupLayout.DEFAULT_SIZE, 1217, Short.MAX_VALUE)
        );
        layout.setVerticalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addComponent(mainScrollPanel, javax.swing.GroupLayout.DEFAULT_SIZE, 871, Short.MAX_VALUE)
        );
    }// </editor-fold>//GEN-END:initComponents

    /**
     * Opens analysis of one selected face, otherwise pops message dialog that
     * you should select just one face
     * @param evt 
     */
    private void analyseButton1MouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_analyseButton1MouseClicked

        if (selectedRows.size() == 1) {

            String name = model.getValueAt(selectedRows.get(0), 1).toString();
            HumanFace face = project.getFaceByName(name);
            createSingleFaceTab(face, name);
        } else {
            JOptionPane.showMessageDialog(this, "Select one model");
        }
    }//GEN-LAST:event_analyseButton1MouseClicked

    /**
     * Inflates models (selected will be deselected and vice versa)
     * @param evt 
     */
    private void inflateButton1MouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_inflateButton1MouseClicked

        for (int i = 0; i < model.getRowCount(); i++) {
            if (model.getValueAt(i, 0) == (Object) true) {
                model.setValueAt(false, i, 0);
            } else {
                model.setValueAt(true, i, 0);
            }
        }
    }//GEN-LAST:event_inflateButton1MouseClicked

    /**
     * Deselects all models from list of models
     * @param evt 
     */
    private void deselectAllButton1MouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_deselectAllButton1MouseClicked
        deselectAllRows();
    }//GEN-LAST:event_deselectAllButton1MouseClicked

    private void deselectAllRows() {
        for (int i = 0; i < model.getRowCount(); i++) {
            model.setValueAt(false, i, 0);
        }
    }
    
    /**
     * Selects all models from list of models
     * @param evt 
     */
    private void selectAllButton1MouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_selectAllButton1MouseClicked
        for (int i = 0; i < model.getRowCount(); i++) {
            model.setValueAt(true, i, 0);
        }
    }//GEN-LAST:event_selectAllButton1MouseClicked

//GEN-FIRST:event_removeButton1MouseClicked
     /**
     * Removes selected models from list and project
     * @param evt Removes selected faces
     */
    private void removeButton1MouseClicked(java.awt.event.MouseEvent evt) {                                           

        removeSelectedFaces();
    }    
//GEN-LAST:event_removeButton1MouseClicked

    private void removeSelectedFaces() {
        Collections.sort(selectedRows, Collections.reverseOrder());
        selectedRows.forEach(row -> {
            HumanFace face = this.project.getFaceByName(model.getValueAt(row, 1).toString());
            this.project.removeFace(face);
            model.removeRow(row);
        });
        selectedRows.clear();        
    }
    
    /**
     * Adds new model
     * @param evt 
     */
    private void addButton1MouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_addButton1MouseClicked
        loadModel();
    }//GEN-LAST:event_addButton1MouseClicked

    /**
     * Shows face state panel after clicking on face in list. Also double-click
     * on face opens Analyze tab for clicked face
     * @param evt 
     */
    private void tableMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_tableMouseClicked
        if (table.getSelectedRow() == -1) {
            infoPanel.setVisible(false);
        } else {
            
            if (evt.getClickCount() == 2) {
                deselectAllRows();
                model.setValueAt(true, table.getSelectedRow(), 0);
                analyseButton1MouseClicked(evt);
            }
            infoPanel.setVisible(true);
            checkFaceState(table.getValueAt(table.getSelectedRow(), 1).toString());       
        }
    }//GEN-LAST:event_tableMouseClicked

    /**
     * Opens 1:1 tab with two selected faces, otherwise pops message that you
     * should select two faces
     * @param evt 
     */
    private void oneOnOneButton1MouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_oneOnOneButton1MouseClicked
        //loadTwoModels();

        if (selectedRows.size() == 2) {

            String name1 = model.getValueAt(selectedRows.get(0), 1).toString();
            String name2 = model.getValueAt(selectedRows.get(1), 1).toString();
            HumanFace face1 = project.getFaceByName(name1);
            HumanFace face2 = project.getFaceByName(name2);
            createFaceToFaceTab(face1, face2, name1 + ":" + name2);
        } else {
            JOptionPane.showMessageDialog(this, "Select two models");
        }
    }//GEN-LAST:event_oneOnOneButton1MouseClicked

    private void manyToManyButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_manyToManyButtonMouseClicked
        createManyToManyTab("N:N");
    }//GEN-LAST:event_manyToManyButtonMouseClicked

    
    /**
     * Updates selectedRows - adds new selected rows or removes deselected rows
     * @param e TableModelEvent
     */
    private void jTable1TableChanged(TableModelEvent e) {
        
        if (e.getType() == javax.swing.event.TableModelEvent.UPDATE) {
            int row = e.getFirstRow();
           
            if (table.getValueAt(row, 0) == (Object)true) {
                if (!selectedRows.contains(row)) {
                    selectedRows.add(row);
                }
            } else {
                if (selectedRows.contains(row)) {
                    selectedRows.remove((Integer)row);
                }   
            }
        }
    } 
   
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JButton addButton1;
    private javax.swing.JButton analyseButton1;
    private javax.swing.JPanel buttonsPanel;
    private javax.swing.JButton deselectAllButton1;
    private javax.swing.JScrollPane faceTableScrollPanel;
    private javax.swing.JPanel filterPanel;
    private javax.swing.JButton inflateButton1;
    private javax.swing.JPanel infoPanel;
    private javax.swing.JPanel mainPanel;
    private javax.swing.JScrollPane mainScrollPanel;
    private javax.swing.JButton manyToManyButton;
    private javax.swing.JButton oneOnOneButton1;
    private javax.swing.JButton removeButton1;
    private javax.swing.JButton selectAllButton1;
    private javax.swing.JTable table;
    // End of variables declaration//GEN-END:variables
    @Override
    public void componentOpened() {
        // TODO add custom code on component opening
    }

    @Override
    public void componentClosed() {
        // TODO add custom code on component closing
    }

    void writeProperties(java.util.Properties p) {
        // better to version settings since initial version as advocated at
        // http://wiki.apidesign.org/wiki/PropertyFiles
        p.setProperty("version", "1.0");
        // TODO store your settings
    }

    void readProperties(java.util.Properties p) {
        String version = p.getProperty("version");
        // TODO read your settings according to their version
    }

    /**
     * Loads model selected in file chooser by user
     */
    public void loadModel() {
        File[] files = new FileChooserBuilder(ProjectTopComp.class)
                .setTitle("Open human face(s)")
                .setDefaultWorkingDirectory(new File(System.getProperty("user.home")))
                //.setApproveText("Add")
                .setFileFilter(new FileNameExtensionFilter("obj files (*.obj)", "obj"))
                .setAcceptAllFileFilterUsed(true)
                .showMultiOpenDialog();
        
        
        if (files == null) {
            System.out.print("No file chosen.");
        } else {
            Logger out = Logger.measureTime();
            for (File file : files) {
                HumanFace face = null;
                try {
                    face = new HumanFace(file);
                    out.printDuration("Loaded model " + face.getShortName() +" with " + face.getMeshModel().getNumVertices() + " vertices");
                    // simple hack:
                    Path path = Paths.get(file.getAbsolutePath());
                    Path folder = path.getParent();
                    Path filename = path.getFileName();
                    String filestr = filename.toString();
                    filestr = filestr.split("_ECA.obj")[0];
                    filestr = filestr + "_landmarks.csv";
                    face.loadFeaturePoints(folder.toString(), filestr);
                } catch (IOException ex) {
                    ex.printStackTrace();
                }

                String name = face.getShortName();
                if (this.project.getFaceByName(name) == null) {
                    this.project.addFace(face);
                    face.registerListener(model);
                    model.addRowWithName(name, false);
                } else {
                    JOptionPane.showMessageDialog(this, String.format("Model with name %s is already loaded", name));
                }
            }
        }
    }

   /**
    * Creates and opens tab with one face
    * @param face which will be analyzed
    * @param name name of the tab (name of the model)
    */
    private void createSingleFaceTab(HumanFace face, String name) {
        SingleFaceTab newTab = new SingleFaceTab(face, name);
        this.singleFaceTabs.put(face, newTab);
        newTab.open();
        newTab.requestActive();
    }

    /**
     * Creates and opens tab with two faces (1:1 analysis)
     * @param face1 which will be analyzed
     * @param face2 which will be analyzed
     * @param name name of the tab
     */
    private void createFaceToFaceTab(HumanFace face1, HumanFace face2, String name) {
        FaceToFaceTab newTab = new FaceToFaceTab(face1, face2, name);
        this.faceToFaceTabs.put(face1, newTab);
        this.faceToFaceTabs.put(face2, newTab);
        newTab.open();
        newTab.requestActive();
    }
    
    /**
     * Creates and opens tab with multiple faces for N:N analysis
     * @param faces faces to be analyzed
     * @param name name of the tab
     */
    private void createManyToManyTab(String name) {
        ManyToManyTab newTab = new ManyToManyTab(name);
        newTab.open();
        newTab.requestActive();
    }

    /**
     * Opens info panel with face state information
     * @param faceName String name of face
     */
    private void checkFaceState(String faceName) {
        HumanFace face = project.getFaceByName(faceName);
        FaceStatePanel fsp = new FaceStatePanel(infoPanel, face);
    }
    
    private void alphabeticalFilter() {
        this.deselectAllRows();
        /*
        TableRowSorter<TableModel> sorter = new TableRowSorter<>(model);
        jTable1.setRowSorter(sorter);
        
        List<RowSorter.SortKey> sortKeys = new ArrayList<>();
        sortKeys.add(new RowSorter.SortKey(1, SortOrder.ASCENDING));
        sorter.setSortKeys(sortKeys);*/
        
        List<String> names = new ArrayList();
        while (model.getRowCount() > 0) {
            names.add(model.getValueAt(0, 1).toString());
            model.removeRow(0);
        }
        Collections.sort(names);
        names.forEach(name -> {
            HumanFace face = project.getFaceByName(name);
            model.addRowWithName(name, face.hasKdTree());
        });
        
    }
    
    /**
     * Removes faces from project (and table of faces) based on filter configuration
     * 
     */
    private void applyFilter() {
        
        deselectAllRows();
                
        for (int i = 0; i < model.getRowCount(); i++) {
            
            HumanFace face = project.getFaceByName(model.getValueAt(i, 1).toString());
            
            if ((fp.isKdTreeFilter() && !face.hasKdTree()) || (fp.isFeaturePointsFilter() && !face.hasFeaturePoints())) {
                model.setValueAt(true, i, 0);
            }
        }
        
        removeSelectedFaces();
        
        if (fp.isAlphaBeticalFilter()) {
            alphabeticalFilter();
        }
    }
    
}
