package cz.fidentis.analyst.batch;

import com.jogamp.opengl.GL2;
import cz.fidentis.analyst.Logger;
import cz.fidentis.analyst.canvas.Canvas;
import cz.fidentis.analyst.core.ProgressDialog;
import cz.fidentis.analyst.face.HumanFace;
import cz.fidentis.analyst.face.HumanFaceFactory;
import cz.fidentis.analyst.icp.IcpTransformer;
import cz.fidentis.analyst.icp.NoUndersampling;
import cz.fidentis.analyst.icp.RandomStrategy;
import cz.fidentis.analyst.icp.UndersamplingStrategy;
import cz.fidentis.analyst.mesh.core.MeshModel;
import cz.fidentis.analyst.scene.DrawableFace;
import cz.fidentis.analyst.visitors.kdtree.AvgFaceConstructor;
import java.io.IOException;
import java.nio.file.Path;
import java.util.List;
import javax.swing.SwingWorker;

/**
 * A task that registers multiple faces using ICP and, simultaneously,
 * computes average face. 
 * The exact computation parameters are taken from the {@code BatchPanel}.
 * 
 * @author Radek Oslejsek
 */
public class IcpTask extends SwingWorker<MeshModel, HumanFace> {

    private HumanFace avgFace;
    private final ProgressDialog progressDialog;
    private final BatchPanel controlPanel;
    private final Canvas canvas;
    private int avgFaceSceneIndex = -1;
    private int icpFaceSceneIndex = -1;
    
    private final Stopwatch totalTime = new Stopwatch("Total computation time:\t");
    private final Stopwatch avgFaceComputationTime = new Stopwatch("AVG face computation time:\t");
    private final Stopwatch icpComputationTime = new Stopwatch("ICP registration time:\t");
    private final Stopwatch loadTime = new Stopwatch("File (re-)loading time:\t");
    private final Stopwatch kdTreeConstructionTime = new Stopwatch("KD trees construction time:\t");
    
    /**
     * Constructor.
     * 
     * @param progressDialog A window that show the progress of the computation. Must not be {@code null}
     * @param controlPanel A control panel with computation parameters. Must not be {@code null}
     * @param canvas A canvas with 3D scene. If not {@code null}, then the transformed faces are rendered one by one.
     */
    public IcpTask(ProgressDialog progressDialog, BatchPanel controlPanel, Canvas canvas) {
        try {
            avgFace = new HumanFace(controlPanel.getFaces().get(controlPanel.getTemplateFaceIndex()).toFile());
        } catch (IOException ex) {
            throw new IllegalArgumentException(ex);
        }

        this.progressDialog = progressDialog;
        this.controlPanel = controlPanel;
        this.canvas = canvas;
    }

    @Override
    protected MeshModel doInBackground() throws Exception {
        HumanFaceFactory factory = controlPanel.getHumanFaceFactory();
        List<Path> faces = controlPanel.getFaces();
        int undersampling = controlPanel.getIcpUndersampling();
        boolean computeICP = controlPanel.computeICP();
        boolean computeAvgFace = controlPanel.computeAvgFace();

        controlPanel.getHumanFaceFactory().setReuseDumpFile(true);
        factory.setStrategy(HumanFaceFactory.Strategy.LRU);
        
        totalTime.start();

        AvgFaceConstructor avgFaceConstructor = new AvgFaceConstructor(avgFace.getMeshModel());

        for (int i = 0; i < faces.size(); i++) {

            if (isCancelled()) {
                return null;
            }

            // Compute AVG template face. Use each tranfromed face only once. Skip the original face
            if (i != controlPanel.getTemplateFaceIndex() && (computeICP || computeAvgFace)) {

                loadTime.start();
                String faceId = factory.loadFace(faces.get(i).toFile());
                HumanFace face = factory.getFace(faceId);
                loadTime.stop();

                kdTreeConstructionTime.start();
                face.computeKdTree(false);
                kdTreeConstructionTime.stop();

                if (computeICP) {// ICP registration:
                    icpComputationTime.start();
                    IcpTransformer icp = new IcpTransformer(
                            avgFace.getMeshModel(), 
                            100, 
                            controlPanel.scaleIcp(), 
                            0.3, 
                            (undersampling == 100) ? new NoUndersampling() : new RandomStrategy(undersampling));
                    face.getMeshModel().compute(icp, true);
                    icpComputationTime.stop();
                }

                if (computeAvgFace) { // AVG template face
                    avgFaceComputationTime.start();
                    face.getKdTree().accept(avgFaceConstructor);
                    avgFaceComputationTime.stop();
                }

                publish(face);
            }

            int progress = (int) Math.round(100.0 * (i + 1.0) / faces.size());
            progressDialog.setValue(progress);
        }

        if (computeAvgFace) {
            avgFace.setMeshModel(avgFaceConstructor.getAveragedMeshModel());
        }
        
        totalTime.stop();

        printTimeStats();

        return avgFace.getMeshModel();
    }

    @Override
    protected void done() {
        progressDialog.dispose(); // close progess bar
        if (isCancelled()) {
            avgFace = null;
        }
        if (canvas != null && avgFaceSceneIndex >= 0) {
            canvas.getScene().setDrawableFace(avgFaceSceneIndex, null); // remove from scene
            avgFaceSceneIndex = -1;
        }
        if (canvas != null && icpFaceSceneIndex >= 0) {
            canvas.getScene().setDrawableFace(icpFaceSceneIndex, null); // remove from scene
            icpFaceSceneIndex = -1;
        }
        if (canvas != null) {
            canvas.getCamera().initLocation();
        }
    }

    @Override
    protected void process(List<HumanFace> chunks) {        
        chunks.stream().forEach(f -> {
            if (canvas != null) {
                if (icpFaceSceneIndex == -1) { // first rendering
                    avgFaceSceneIndex = canvas.getScene().getFreeIndex();
                    
                    // locate the camera to the best angle:
                    canvas.getCamera().initLocation(); 
                    canvas.getCamera().rotate(10, -80);
                    canvas.getCamera().move(40, 20);
                    
                    canvas.getScene().setDrawableFace(avgFaceSceneIndex, avgFace);
                    canvas.getScene().getDrawableFace(avgFaceSceneIndex).setRenderMode(GL2.GL_POINT);
                    canvas.getScene().getDrawableFace(avgFaceSceneIndex).setTransparency(0.7f);
                    
                    icpFaceSceneIndex = canvas.getScene().getFreeIndex();                    
                }
                
                canvas.getScene().setDrawableFace(icpFaceSceneIndex, f);
                canvas.getScene().getDrawableFace(icpFaceSceneIndex).setTransparency(0.5f);
                canvas.getScene().getDrawableFace(icpFaceSceneIndex).setColor(DrawableFace.SKIN_COLOR_SECONDARY);
                canvas.renderScene();
            }
        });
    }

    public HumanFace getAverageFace() {
        return this.avgFace;
    }

    protected void printTimeStats() {
        Logger.print(avgFaceComputationTime.toString());
        Logger.print(icpComputationTime.toString());
        Logger.print(loadTime.toString());
        Logger.print(kdTreeConstructionTime.toString());
        Logger.print(totalTime.toString());
    }
}
