package cz.fidentis.analyst.symmetry;

import cz.fidentis.analyst.canvas.Canvas;
import cz.fidentis.analyst.core.ControlPanelAction;
import cz.fidentis.analyst.scene.DrawablePlane;

import java.awt.event.ActionEvent;
import javax.swing.JTabbedPane;
import javax.swing.JToggleButton;

/**
 * Action listener for the manipulation with the symmetry plane.
 * 
 * @author Radek Oslejsek
 */
public class SymmetryAction extends ControlPanelAction {

    private final SymmetryPanel controlPanel;
    private final JTabbedPane topControlPanel;

    /**
     * Constructor.
     * 
     * @param canvas OpenGL canvas
     * @param topControlPanel Top component for placing control panels
     */
    public SymmetryAction(Canvas canvas, JTabbedPane topControlPanel) {
        super(canvas, topControlPanel);
        this.topControlPanel = topControlPanel;
        this.controlPanel = new SymmetryPanel(this);

        // Place control panel to the topControlPanel
        this.topControlPanel.addTab(controlPanel.getName(), controlPanel.getIcon(), controlPanel);
        this.topControlPanel.addChangeListener(e -> {
            // If the symmetry panel is focused...
            if (((JTabbedPane) e.getSource()).getSelectedComponent() instanceof SymmetryPanel) {
                getCanvas().getScene().setDefaultColors();

                if (controlPanel.isShowPlaneChecked()) {
                    getCanvas().getScene().showSymmetryPlanes();
                }
            } else {
                getCanvas().getScene().hideSymmetryPlanes();
            }
        });

        recomputeSymmetryPlane();
    }
    
    @Override
    public void actionPerformed(ActionEvent ae) {
        String action = ae.getActionCommand();
        
        switch (action) {
            case SymmetryPanel.ACTION_COMMAND_SHOW_HIDE_PANEL:
                hideShowPanelActionPerformed(ae, this.controlPanel);   
                break;
            case SymmetryPanel.ACTION_COMMAND_SHOW_HIDE_PLANE:
                if (((JToggleButton) ae.getSource()).isSelected()) {
                    getScene().showSymmetryPlanes();
                } else {
                    getScene().hideSymmetryPlanes();
                }
                break;
            case SymmetryPanel.ACTION_COMMAND_RECOMPUTE: 
                recomputeSymmetryPlane();
                break;  
            default:
                // do nothing
        }
        renderScene();
    }
    
    private void recomputeSymmetryPlane() {
        SymmetryEstimator primaryEstimator = new SymmetryEstimator(controlPanel.getSymmetryConfig());
        getPrimaryDrawableFace().getModel().compute(primaryEstimator);

        DrawablePlane primarySymmetryPlane = new DrawablePlane(primaryEstimator.getSymmetryPlaneMesh(),
                primaryEstimator.getSymmetryPlane());
        primarySymmetryPlane.setTransparency(0.5f);
        getScene().setDrawableSymmetryPlane(0, primarySymmetryPlane);

        DrawablePlane primaryCuttingPlane = new DrawablePlane(primaryEstimator.getSymmetryPlaneMesh(),
                primaryEstimator.getSymmetryPlane());
        primaryCuttingPlane.setTransparency(0.5f);
        getScene().setDrawableCuttingPlane(0, primaryCuttingPlane);

        DrawablePlane primaryMirrorPlane = new DrawablePlane(primaryEstimator.getSymmetryPlaneMesh(),
                primaryEstimator.getSymmetryPlane());
        primaryMirrorPlane.setTransparency(0.5f);
        getScene().setDrawableMirrorPlane(0, primaryMirrorPlane);

        if (getSecondaryDrawableFace() != null) {
            SymmetryEstimator secondaryEstimator = new SymmetryEstimator(controlPanel.getSymmetryConfig());
            getSecondaryDrawableFace().getModel().compute(secondaryEstimator);

            DrawablePlane secondarySymmetryPlane = new DrawablePlane(secondaryEstimator.getSymmetryPlaneMesh(),
                    secondaryEstimator.getSymmetryPlane());
            secondarySymmetryPlane.setTransparency(0.5f);
            getScene().setDrawableSymmetryPlane(1, secondarySymmetryPlane);

            DrawablePlane secondaryCuttingPlane = new DrawablePlane(secondaryEstimator.getSymmetryPlaneMesh(),
                    secondaryEstimator.getSymmetryPlane());
            secondaryCuttingPlane.setTransparency(0.5f);
            getScene().setDrawableCuttingPlane(1, secondaryCuttingPlane);

            DrawablePlane secondaryMirrorPlane = new DrawablePlane(secondaryEstimator.getSymmetryPlaneMesh(),
                    secondaryEstimator.getSymmetryPlane());
            secondaryMirrorPlane.setTransparency(0.5f);
            getScene().setDrawableMirrorPlane(1, secondaryMirrorPlane);
        }

        getCanvas().getScene().hideSymmetryPlanes();
    }
    
}
