package cz.fidentis.analyst.registration;

import cz.fidentis.analyst.Logger;
import cz.fidentis.analyst.canvas.Canvas;
import cz.fidentis.analyst.core.ControlPanelAction;
import cz.fidentis.analyst.face.HumanFace;
import cz.fidentis.analyst.face.HumanFaceUtils;
import cz.fidentis.analyst.face.events.HausdorffDistanceComputed;
import cz.fidentis.analyst.face.events.HumanFaceEvent;
import cz.fidentis.analyst.face.events.HumanFaceListener;
import cz.fidentis.analyst.face.events.HumanFaceTransformedEvent;
import cz.fidentis.analyst.procrustes.ProcrustesAnalysis;
import java.awt.Color;
import java.awt.event.ActionEvent;
import java.util.zip.DataFormatException;
import javax.swing.JCheckBox;
import javax.swing.JFormattedTextField;
import javax.swing.JOptionPane;
import javax.swing.JTabbedPane;
import javax.vecmath.Vector3d;

/**
 * Action listener for the ICP and Procrustes registration.
 * <p>
 * Besides the UX logic, this object stores the parameters and results of 
 * manual, ICP, or Procrustes registration (alignment of human faces) 
 * <p/>
 * <p>
 * This object also serves as {@code HumanFaceListener}.
 * It means that it is invoked whenever one of the faces are changed and then can 
 * react to these changes.
 * <p/>
 * <p>
 * Changes made by this objects are announced to other listeners. 
 * Following events are triggered:
 * <ul>
 * <li>{@code MeshChangedEvent}</li>
 * <li></li>
 * </ul>
 * </p>
 *
 * @author Richard Pajersky
 * @author Radek Oslejsek
 * @author Daniel Schramm
 */
public class RegistrationAction extends ControlPanelAction implements HumanFaceListener  {

    /*
     * Attributes handling the state
     */
    private boolean scale = true;
    private int maxIterations = 10;
    private double error = 0.3;
    private String strategy = RegistrationPanel.STRATEGY_POINT_TO_POINT;
    private boolean relativeDist = false;
    private boolean heatmapRender = false;
    private boolean procrustesScalingEnabled = false;

    /*
     * Coloring threshold and statistical values of feature point distances:
     */
    private double fpThreshold = 5.0;

    private final RegistrationPanel controlPanel;

    /**
     * Constructor.
     *
     * @param canvas OpenGL canvas
     * @param topControlPanel Top component for placing control panels
     */
    public RegistrationAction(Canvas canvas, JTabbedPane topControlPanel) {
        super(canvas, topControlPanel);
        this.controlPanel = new RegistrationPanel(this);

        // Place control panel to the topControlPanel
        topControlPanel.addTab(controlPanel.getName(), controlPanel.getIcon(), controlPanel);
        topControlPanel.addChangeListener(e -> {
            // If the registration panel is focused...
            if (((JTabbedPane) e.getSource()).getSelectedComponent() instanceof RegistrationPanel) {
                // ... display heatmap and feature points relevant to the registration
                //getCanvas().getScene().setDefaultColors();
                highlightCloseFeaturePoints();
                //getSecondaryDrawableFace().setRenderHeatmap(heatmapRender);
            }
        });
        topControlPanel.setSelectedComponent(controlPanel); // Focus registration panel
        //calculateHausdorffDistance();
        highlightCloseFeaturePoints();
        
        // Be informed about changes in faces perfomed by other GUI elements
        getPrimaryDrawableFace().getHumanFace().registerListener(this);
        getSecondaryDrawableFace().getHumanFace().registerListener(this);
    }

    @Override
    public void actionPerformed(ActionEvent ae) {
        double value;
        String action = ae.getActionCommand();

//        OutputWindow.print(ae.getActionCommand());
        switch (action) {
            case RegistrationPanel.ACTION_COMMAND_APPLY_ICP:
                applyICP();
                highlightCloseFeaturePoints();
                //announceMeshChange(getSecondaryDrawableFace());
                getCanvas().getSecondaryFace().announceEvent(new HumanFaceTransformedEvent(
                        getCanvas().getSecondaryFace(), "", this)
                );
                break;
            case RegistrationPanel.ACTION_COMMAND_MANUAL_TRANSFORMATION_IN_PROGRESS:
                HumanFaceUtils.transformFace(
                        getSecondaryDrawableFace().getHumanFace(),
                        controlPanel.getAndClearManualRotation(),
                        controlPanel.getAndClearManualTranslation(),
                        controlPanel.getAndClearManualScale(),
                        false
                );
                highlightCloseFeaturePoints();
                getCanvas().getSecondaryFace().announceEvent(new HumanFaceTransformedEvent(
                        getCanvas().getSecondaryFace(), "", this, false) // transformation under progress
                );
                break;
            case RegistrationPanel.ACTION_COMMAND_MANUAL_TRANSFORMATION_FINISHED:
                getCanvas().getSecondaryFace().announceEvent(new HumanFaceTransformedEvent(
                        getCanvas().getSecondaryFace(), "", this, true) // finished transformation
                );
                //announceMeshChange(getSecondaryDrawableFace());
                break;
            case RegistrationPanel.ACTION_COMMAND_FP_CLOSENESS_THRESHOLD:
                fpThreshold = ((Number) (((JFormattedTextField) ae.getSource()).getValue())).doubleValue();
                highlightCloseFeaturePoints();
                break;
            case RegistrationPanel.ACTION_COMMAND_ICP_SCALE:
                this.scale = ((JCheckBox) ae.getSource()).isSelected();
                break;
            case RegistrationPanel.ACTION_COMMAND_ICP_MAX_ITERATIONS:
                maxIterations = ((Number) (((JFormattedTextField) ae.getSource()).getValue())).intValue();
                break;
            case RegistrationPanel.ACTION_COMMAND_ICP_ERROR:
                error = ((Number) (((JFormattedTextField) ae.getSource()).getValue())).doubleValue();
                break;
            case RegistrationPanel.ACTION_COMMAND_PROCRUSTES_SCALE:
                this.procrustesScalingEnabled = ((JCheckBox) ae.getSource()).isSelected();
                break;
            case RegistrationPanel.ACTION_COMMAND_PROCRUSTES_APPLY:
                applyProcrustes();
                highlightCloseFeaturePoints();
                getCanvas().getPrimaryFace().announceEvent(new HumanFaceTransformedEvent(
                        getCanvas().getPrimaryFace(), "", this)
                );
                getCanvas().getSecondaryFace().announceEvent(new HumanFaceTransformedEvent(
                        getCanvas().getSecondaryFace(), "", this)
                );
                //announceMeshChange(getPrimaryDrawableFace());
                //announceMeshChange(getSecondaryDrawableFace());
                break;
            case RegistrationPanel.ACTION_COMMAND_ALIGN_SYMMETRY_PLANES:
                alignSymmetryPlanes();
                highlightCloseFeaturePoints();
                getCanvas().getSecondaryFace().announceEvent(new HumanFaceTransformedEvent(
                        getCanvas().getSecondaryFace(), "", this)
                );
                //announceMeshChange(getSecondaryDrawableFace());
                break;
            default:
                // do nothing
        }

        renderScene();
    }
    
    @Override
    public void acceptEvent(HumanFaceEvent event) {
        if (event.getIssuer() == this) {
            return;
        }
        
        if (event instanceof HausdorffDistanceComputed) { // update stats
            HausdorffDistanceComputed hdEvent = (HausdorffDistanceComputed) event;
            controlPanel.updateHausdorffDistanceStats(
                    hdEvent.getHusdorffDistStats(),
                    hdEvent.getWeightedHusdorffDistStats()
            );
        }
    }
    
    /**
     * Calculates Procrustes analysis.
     * 
     * First it creates object of type ProcrusteAnalasys containing two face models converted
     * to ProcrustesAnalysisFaceModel objects which are required for next analysis step.
     * 
     * In the analysis step, faces are superimposed and rotated.
     * 
     * If the {@code procrustesScalingEnabled} attribute is set to true, then one of the faces is scaled as well.
     */
    protected void applyProcrustes() {
        Logger out = Logger.measureTime();

        HumanFace primaryFace = getCanvas().getPrimaryFace();
        HumanFace secondaryFace = getCanvas().getSecondaryFace();

        try {
            ProcrustesAnalysis procrustesAnalysisInit = new ProcrustesAnalysis(primaryFace, secondaryFace, this.procrustesScalingEnabled);
            procrustesAnalysisInit.analyze();
            primaryFace.removeKdTree(); // invalidate k-d tree, if exists
            secondaryFace.removeKdTree(); // invalidate k-d tree, if exists
            highlightCloseFeaturePoints();
        } catch (DataFormatException e) {
            Logger.print("Procrustes Analysis experienced exception");
        }

        out.printDuration("Procrustes for models with "
                + getPrimaryDrawableFace().getHumanFace().getMeshModel().getNumVertices()
                + "/"
                + getSecondaryDrawableFace().getHumanFace().getMeshModel().getNumVertices()
                + " vertices and "
                + getPrimaryDrawableFace().getHumanFace().getFeaturePoints().size()
                + " feature points"
        );
    }
    
    protected void applyICP() {
        Logger out = Logger.measureTime();
        
        HumanFaceUtils.alignMeshes(
                getPrimaryDrawableFace().getHumanFace(),
                getSecondaryDrawableFace().getHumanFace(), // is transformed
                maxIterations,
                scale,
                error,
                controlPanel.getIcpUndersampling(),
                false // drop k-d tree, if exists
        );
        
        out.printDuration("ICP for models with "
                + getPrimaryDrawableFace().getHumanFace().getMeshModel().getNumVertices()
                + "/"
                + getSecondaryDrawableFace().getHumanFace().getMeshModel().getNumVertices()
                + " vertices"
        );
    }
    
    /**
     * Calculates feature points which are too far away and changes their color
     * to green otherwise set color to default
     */
    private void highlightCloseFeaturePoints() {
        if (getPrimaryDrawableFace() == null) { // scene not yet initiated
            return;
        }

        if (getPrimaryFeaturePoints() == null
                || getSecondaryFeaturePoints() == null
                || getPrimaryFeaturePoints().getFeaturePoints().size() != getSecondaryFeaturePoints().getFeaturePoints().size()) {
            return;
        }

        double fpMaxDist = Double.NEGATIVE_INFINITY;
        double fpMinDist = Double.POSITIVE_INFINITY;
        double distSum = 0.0;
        for (int i = 0; i < getPrimaryFeaturePoints().getFeaturePoints().size(); i++) {
            Vector3d v = new Vector3d(getPrimaryFeaturePoints().getFeaturePoints().get(i).getPosition());
            v.sub(getSecondaryFeaturePoints().getFeaturePoints().get(i).getPosition());
            double distance = v.length();
            if (distance > fpThreshold) {
                getPrimaryFeaturePoints().resetColorToDefault(i);
                getSecondaryFeaturePoints().resetColorToDefault(i);
            } else {
                getPrimaryFeaturePoints().setColor(i, Color.GREEN);
                getSecondaryFeaturePoints().setColor(i, Color.GREEN);
            }
            fpMaxDist = Math.max(fpMaxDist, distance);
            fpMinDist = Math.min(fpMinDist, distance);
            distSum += distance;
        }
        double fpAvgDist = distSum / getPrimaryFeaturePoints().getFeaturePoints().size();
        // to do: show ACF dist
    }

    //protected void announceMeshChange(DrawableFace face) {
    //    if (face != null) {
    //        face.getHumanFace().announceEvent(new MeshChangedEvent(face.getHumanFace(), face.getHumanFace().getShortName(), this));
    //    }
    //}

    private void alignSymmetryPlanes() {
        if (!getPrimaryDrawableFace().getHumanFace().hasSymmetryPlane() 
                || !getSecondaryDrawableFace().getHumanFace().hasSymmetryPlane()) {
            
            JOptionPane.showMessageDialog(controlPanel,
                    "Compute symmetry planes first",
                    "No symmerty planes",
                    JOptionPane.INFORMATION_MESSAGE);
            return;
        }
        
        HumanFaceUtils.alignSymmetryPlanes(
                getPrimaryDrawableFace().getHumanFace(),
                getSecondaryDrawableFace().getHumanFace(), // is transformed
                false // drops k-d tree, if exists
        );
    }
}
