package cz.fidentis.analyst.distance;

import com.jogamp.opengl.GL2;
import cz.fidentis.analyst.Logger;
import cz.fidentis.analyst.canvas.Canvas;
import cz.fidentis.analyst.core.LoadedActionEvent;
import cz.fidentis.analyst.core.ControlPanelAction;
import cz.fidentis.analyst.face.events.HausdorffDistanceComputed;
import cz.fidentis.analyst.face.events.HumanFaceEvent;
import cz.fidentis.analyst.face.events.HumanFaceListener;
import cz.fidentis.analyst.face.events.HumanFaceTransformedEvent;
import cz.fidentis.analyst.feature.FeaturePoint;
import cz.fidentis.analyst.feature.FeaturePointType;
import cz.fidentis.analyst.scene.DrawableFpWeights;
import cz.fidentis.analyst.visitors.face.HausdorffDistancePrioritized;
import cz.fidentis.analyst.visitors.mesh.HausdorffDistance;
import cz.fidentis.analyst.visitors.mesh.HausdorffDistance.Strategy;
import java.awt.Color;
import java.awt.event.ActionEvent;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JTabbedPane;
import javax.swing.JToggleButton;

/**
 * Action listener for computation of the Hausdorff distance.
 * <p>
 * Besides the UX logic, this object stores the parameters and results of 
 * Hausdorff distance and provide them to other parts of GUI. 
 * <p/>
 * <p>
 * This object also serves as {@code HumanFaceListener}.
 * It means that it is invoked whenever one of the faces are changed and then can 
 * react to these changes, e.g., the distance automatically recomputed whenever 
 * the faces are transformed.
 * <p/>
 * <p>
 * Changes made by this objects are announced to other listeners. 
 * Following events are triggered:
 * <ul>
 * <li>{@code HausdorffDistanceComputed}</li>
 * <li></li>
 * </ul>
 * </p>
 * 
 * @author Radek Oslejsek
 * @author Daniel Schramm
 */
public class DistanceAction extends ControlPanelAction implements HumanFaceListener {
    
    /*
     * Attributes handling the state
     */
    private HausdorffDistancePrioritized whdVisitor = null;
    private HausdorffDistance hdVisitor = null;
    
    private final Map<FeaturePointType, Double> featurePointTypes;
    private String strategy = DistancePanel.SEL_STRATEGY_POINT_TO_TRIANGLE;
    private boolean relativeDist = false;
    private String heatmapDisplayed = DistancePanel.SEL_STD_HAUSDORFF_DISTANCE;
    private String weightedFPsShow = DistancePanel.SEL_SHOW_NO_SPHERES;
    private boolean crop = true;
    
    private FeaturePointType hoveredFeaturePoint = null;
    
    private final DistancePanel controlPanel;
    private final DrawableFpWeights fpSpheres;
    
    private static final Color  FEATURE_POINT_HIGHLIGHT_COLOR = Color.MAGENTA;
    private static final Color  FEATURE_POINT_HOVER_COLOR = Color.CYAN;
    
    /**
     * Constructor.
     * 
     * @param canvas OpenGL canvas
     * @param topControlPanel Top component for placing control panels
     */
    public DistanceAction(Canvas canvas, JTabbedPane topControlPanel) {
        super(canvas, topControlPanel);
        
        // Pre-set all feature points with default weight (spehere size)
        featurePointTypes = getSecondaryFeaturePoints().getFeaturePoints()
                .stream()
                .collect(Collectors.toMap(
                        FeaturePoint::getFeaturePointType,
                        featurePoint -> DrawableFpWeights.FPW_DEFAULT_SIZE));
        
        // Create the panel with initially computed values
        this.controlPanel = new DistancePanel(this, getSecondaryFeaturePoints().getFeaturePoints(), featurePointTypes.keySet());
        
        // Add spheres to the scene
        fpSpheres = new DrawableFpWeights(getSecondaryFeaturePoints().getFeaturePoints());
        getCanvas().getScene().setOtherDrawable(getCanvas().getScene().getFreeSlotForOtherDrawables(), fpSpheres);
        
        // Place control panel to the topControlPanel
        topControlPanel.addTab(controlPanel.getName(), controlPanel.getIcon(), controlPanel);
        topControlPanel.addChangeListener(e -> {
            if (((JTabbedPane) e.getSource()).getSelectedComponent() instanceof DistancePanel) {  // If the distance panel is focused...
                // ... display heatmap and feature points relevant to the Hausdorff distance
                getCanvas().getScene().setDefaultColors();
                setVisibleSpheres();
            } else {
                fpSpheres.show(false);
                //getSecondaryDrawableFace().clearHeatMapSaturation();
            }
        });
        
        // Init the panel, i.e., compute weighted HD with significant points and update GUI:
        actionPerformed(new ActionEvent(this, 0, DistancePanel.ACTION_COMMAND_FEATURE_POINT_SELECT_AUTO));
        fpSpheres.show(false);
        
        // Be informed about changes in faces perfomed by other GUI elements
        getPrimaryDrawableFace().getHumanFace().registerListener(this);
        getSecondaryDrawableFace().getHumanFace().registerListener(this);
    }

    @Override
    public void actionPerformed(ActionEvent ae) {
        final String action = ae.getActionCommand();
        
        List<FeaturePoint> featurePoints = getSecondaryFeaturePoints().getFeaturePoints();
        
        switch (action) {
            case DistancePanel.ACTION_COMMAND_SET_DISPLAYED_HEATMAP:
                heatmapDisplayed = (String) ((JComboBox) ae.getSource()).getSelectedItem();
                updateHeatmap();
                break;
            case DistancePanel.ACTION_COMMAND_SHOW_HIDE_WEIGHTED_FPOINTS:
                weightedFPsShow = (String) ((JComboBox) ae.getSource()).getSelectedItem();
                setVisibleSpheres();
                break;
            case DistancePanel.ACTION_COMMAND_SET_DISTANCE_STRATEGY:
                strategy = (String) ((JComboBox) ae.getSource()).getSelectedItem();
                computeAndUpdateHausdorffDistance(true);
                break;
            case DistancePanel.ACTION_COMMAND_RELATIVE_ABSOLUTE_DIST:
                this.relativeDist = ((String) ((JComboBox) ae.getSource()).getSelectedItem())
                        .equals(DistancePanel.SEL_RELATIVE_DISTANCE);
                computeAndUpdateHausdorffDistance(true);
                break;
            case FeaturePointsPanel.ACTION_COMMAND_FEATURE_POINT_SELECT:
                selectFeaturePoint((LoadedActionEvent) ae);
                setVisibleSpheres();
                highlightSelectedSpheres();
                computeAndUpdateHausdorffDistance(false);
                break;
            case DistancePanel.ACTION_COMMAND_FEATURE_POINT_SELECT_ALL:
                for (int i = 0; i < featurePoints.size(); i++) {
                    featurePointTypes.put(featurePoints.get(i).getFeaturePointType(), fpSpheres.getSize(i));
                }
                controlPanel.getFeaturePointsListPanel().selectAllFeaturePoints(true);
                setVisibleSpheres();
                highlightSelectedSpheres();
                computeAndUpdateHausdorffDistance(false);
                break;
            case DistancePanel.ACTION_COMMAND_FEATURE_POINT_SELECT_NONE:
                featurePointTypes.clear();
                controlPanel.getFeaturePointsListPanel().selectAllFeaturePoints(false);
                setVisibleSpheres();
                highlightSelectedSpheres();
                computeAndUpdateHausdorffDistance(false);
                break;
            case DistancePanel.ACTION_COMMAND_FEATURE_POINT_SELECT_AUTO:
                for (int i = 0; i < featurePoints.size(); i++) { // select all
                    featurePointTypes.put(featurePoints.get(i).getFeaturePointType(), fpSpheres.getSize(i));
                }
                calculateHausdorffDistance(featurePointTypes, false);
                selectSignificantFPs();
                for (int i = 0; i < featurePoints.size(); i++) { // update the check list
                    controlPanel.getFeaturePointsListPanel().selectFeaturePoint(
                            i,
                            featurePointTypes.containsKey(featurePoints.get(i).getFeaturePointType())
                    );
                }
                setVisibleSpheres();
                highlightSelectedSpheres();
                computeAndUpdateHausdorffDistance(false);
                break;
            case DistancePanel.ACTION_COMMAND_HD_AUTO_CROP:
                this.crop = ((JCheckBox) ae.getSource()).isSelected();
                computeAndUpdateHausdorffDistance(true);
                break;
            case FeaturePointsPanel.ACTION_COMMAND_FEATURE_POINT_HOVER_IN:
                hoverFeaturePoint((LoadedActionEvent) ae, true);
                break;
            case FeaturePointsPanel.ACTION_COMMAND_FEATURE_POINT_HOVER_OUT:
                hoverFeaturePoint((LoadedActionEvent) ae, false);
                break;
            case FeaturePointsPanel.ACTION_COMMAND_FEATURE_POINT_RESIZE:
                resizeFeaturePoint((LoadedActionEvent) ae);
                break;
            case FeaturePointsPanel.ACTION_COMMAND_DISTANCE_RECOMPUTE:
                computeAndUpdateHausdorffDistance(true);
                break;
            default:
                // to nothing
        }
        renderScene();
    }
    
    @Override
    public void acceptEvent(HumanFaceEvent event) {
        if (event instanceof HumanFaceTransformedEvent && event.getIssuer() != this) { // recompte (W)HD
            HumanFaceTransformedEvent ftEvent = (HumanFaceTransformedEvent) event;
            if (ftEvent.isFinished()) {
                hdVisitor = null;
                whdVisitor = null;
                computeAndUpdateHausdorffDistance(true);
                // Relocate weight speheres:
                final List<FeaturePoint> secondaryFPs = getSecondaryFeaturePoints().getFeaturePoints();
                final List<FeaturePoint> weightedFPs = fpSpheres.getFeaturePoints();
                for (int i = 0; i < secondaryFPs.size(); i++) {
                    weightedFPs.get(i).getPosition().set(secondaryFPs.get(i).getPosition());
                }
            }
        } else if (event instanceof HausdorffDistanceComputed) { // update stats
            HausdorffDistanceComputed hdEvent = (HausdorffDistanceComputed) event;
            controlPanel.updateHausdorffDistanceStats(
                    hdEvent.getHusdorffDistStats(),
                    hdEvent.getWeightedHusdorffDistStats()
            );
        }
    }
    
    /**
     * Recalculates the Hausdorff distance and updates all GUI elements
     * of {@link DistancePanel}.
     */
    private void computeAndUpdateHausdorffDistance(boolean recomputeHD) {
        calculateHausdorffDistance(featurePointTypes, recomputeHD);
        setFeaturePointWeigths(); // Updates GUI elements that display the calculated Hausdorff distance metrics
        updateHeatmap();
    }
    
    /**
     * (Re)calculates the Hausdorff distance and updates the heat map of the secondary face
     * as well as values of all appropriate GUI elements of {@link DistancePanel}.
     */
    private void updateHeatmap() {
        switch (heatmapDisplayed) {
            case DistancePanel.SEL_STD_HAUSDORFF_DISTANCE:
                getSecondaryDrawableFace().clearHeatMapSaturation();
                break;
            case DistancePanel.SEL_WEIGHTED_HAUSDORFF_DISTANCE:
                getSecondaryDrawableFace().setHeatMapSaturation(
                        whdVisitor.getMergedPriorities()
                                .get(getSecondaryDrawableFace().getHumanFace())
                );
                break;
            default:
                throw new UnsupportedOperationException(heatmapDisplayed);
        }
        getSecondaryDrawableFace().setHeatMap(whdVisitor.getDistances());
    }
    
    /**
     * Re-calculates the Hausdorff distance.
     * 
     * @param featurePoints Types of feature points according to which the Hausdorff
     *                      distances will be prioritized together with
     *                      the radii of priority spheres around the feature
     *                      points of the corresponding type.
     *                      Must not be {@code null}.
     */
    private void calculateHausdorffDistance(Map<FeaturePointType, Double> featurePoints, boolean recomputeHD) {
        final Strategy useStrategy;
        switch (strategy) {
            case DistancePanel.SEL_STRATEGY_POINT_TO_POINT:
                useStrategy = Strategy.POINT_TO_POINT;
                break;
            case DistancePanel.SEL_STRATEGY_POINT_TO_TRIANGLE:
                useStrategy = Strategy.POINT_TO_TRIANGLE_APPROXIMATE;
                break;
            default:
                throw new UnsupportedOperationException(strategy);
        }
        
        if (hdVisitor == null || recomputeHD) {
            getPrimaryDrawableFace().getHumanFace().computeKdTree(false); // recompute if does not exist
            hdVisitor = new HausdorffDistance(
                    getPrimaryDrawableFace().getHumanFace().getKdTree(), 
                    useStrategy, 
                    relativeDist, 
                    true, // parallel
                    crop
            );
        }
        whdVisitor = new HausdorffDistancePrioritized(hdVisitor, featurePoints);
        
        Logger out = Logger.measureTime();
        
        getSecondaryDrawableFace().getHumanFace().accept(whdVisitor);
        
        out.printDuration("Weighted Hausdorff distance for models with " 
                + getPrimaryDrawableFace().getHumanFace().getMeshModel().getNumVertices()
                + "/"
                + getSecondaryDrawableFace().getHumanFace().getMeshModel().getNumVertices()
                + " vertices"
        );
        
        // announce that the new HD from the secondary to the primary face has been computed:
        getSecondaryDrawableFace().getHumanFace().announceEvent(new HausdorffDistanceComputed(
                getSecondaryDrawableFace().getHumanFace(),
                getPrimaryDrawableFace().getHumanFace(),
                this.whdVisitor,
                getSecondaryDrawableFace().getHumanFace().getShortName(),
                this
        ));
   }
    
    /**
     * Updates the GUI elements of {@link DistancePanel} that display
     * the weights of feature points used to calculate the weighted Hausdorff distance.
     */
    private void setFeaturePointWeigths() {
        controlPanel.getFeaturePointsListPanel().updateFeaturePointWeights(
                whdVisitor.getFeaturePointWeights()
                        .get(getSecondaryDrawableFace().getHumanFace()) // Get FP weights for the secondary face
                        .entrySet()
                        .stream()
                        .collect(
                                Collectors.toMap(
                                        Map.Entry::getKey,            // For each FP type at the secondary face...
                                        weights -> weights.getValue() // ... compute average FP weight over all its facets
                                                .values()
                                                .stream()
                                                .mapToDouble(Double::doubleValue)
                                                .average()
                                                .orElse(Double.NaN))));
    }

    private void selectFeaturePoint(LoadedActionEvent actionEvent) {
        final int index = (int) actionEvent.getData();
        final FeaturePointType fpType = getTypeOfFeaturePoint(index);
        
        if (((JToggleButton) actionEvent.getSource()).isSelected()) {
            featurePointTypes.put(fpType, fpSpheres.getSize(index));
        } else {
            featurePointTypes.remove(fpType);
        }
    }
    
    /**
     * Changes the color of the secondary face's feature point at the given index
     * and of its weighted representation when the cursor hovers over the feature point's name.
     * The index is received as the data payload of {@code actionEvent}.
     * 
     * @param actionEvent Action event with the index of the feature point as its payload data
     * @param entered {@code true} if the cursor entered the feature point,
     *                {@code false} if the cursor left the feature point
     */
    private void hoverFeaturePoint(LoadedActionEvent actionEvent, boolean entered) {
        final int index = (int) actionEvent.getData();
        
        if (entered) { // entering a feature point
            fpSpheres.setColor(index, FEATURE_POINT_HOVER_COLOR);
            getSecondaryFeaturePoints().setColor(index, FEATURE_POINT_HOVER_COLOR);
            hoveredFeaturePoint = getTypeOfFeaturePoint(index);
        } else if (featurePointTypes.containsKey(hoveredFeaturePoint)) { // leaving highlighted FP
            fpSpheres.setColor(index, FEATURE_POINT_HIGHLIGHT_COLOR);
            getSecondaryFeaturePoints().resetColorToDefault(index);
        } else { // leaving ordinary FP
            fpSpheres.resetColorToDefault(index);
            getSecondaryFeaturePoints().resetColorToDefault(index);
        }
    }

    /**
     * Changes the size of the secondary face's feature point at the given index.
     * The index is received as the data payload of {@code actionEvent}.
     * 
     * @param actionEvent Action event with the index of the feature point as its payload data
     */
    private void resizeFeaturePoint(LoadedActionEvent actionEvent) {
        final int index = (int) actionEvent.getData();
        final double size = this.controlPanel.getFeaturePointsListPanel().getSphereSize(index);
        
        fpSpheres.setSize(index, size);
        featurePointTypes.replace(getTypeOfFeaturePoint(index), size);
    }
    
    /**
     * Returns type of the feature point at the given index in the secondary face.
     * 
     * @param index Index of the feature point
     * @return Type of the feature point or {@code null}
     */
    private FeaturePointType getTypeOfFeaturePoint(int index) {
        final List<FeaturePoint> featurePoints = getSecondaryFeaturePoints().getFeaturePoints();
        if (index < 0 || index >= featurePoints.size()) {
            return null;
        }
        
        return featurePoints.get(index)
                .getFeaturePointType();
    }
    
    private void highlightSelectedSpheres() {
        final List<FeaturePoint> secondaryFPs = getSecondaryFeaturePoints().getFeaturePoints();
        if (secondaryFPs == null) {
            return;
        }
        for (int i = 0; i < secondaryFPs.size(); i++) {
            if (featurePointTypes.containsKey(secondaryFPs.get(i).getFeaturePointType())) {
                fpSpheres.setColor(i, FEATURE_POINT_HIGHLIGHT_COLOR);
            } else {
                fpSpheres.resetColorToDefault(i);
            }
        }
    }
    
    private void selectSignificantFPs() {
        // Compute a map of FPs with non-NaN weight
        Map<FeaturePointType, Double> wMap = whdVisitor.getFeaturePointWeights()
                .get(getSecondaryDrawableFace().getHumanFace()) // Get FP weights for the secondary face
                .entrySet()
                .stream()
                .map(fpWeights -> Map.entry(
                        fpWeights.getKey(),  // For each FP type at the secondary face...
                        fpWeights.getValue() // ... compute average FP weight over all its facets
                                .values()
                                .stream()
                                .mapToDouble(Double::doubleValue)
                                .average()
                                .orElse(Double.NaN)))
                .filter(fpWeight -> !Double.isNaN(fpWeight.getValue())) // Filter out feature points with Double.NaN weight
                .collect(Collectors.toMap(
                        Map.Entry::getKey,
                        fpType -> DrawableFpWeights.FPW_DEFAULT_SIZE));
        
        List<FeaturePoint> featurePoints = getSecondaryFeaturePoints().getFeaturePoints();
        featurePointTypes.clear();
        for (int i = 0; i < featurePoints.size(); i++) {
            FeaturePointType fpt = featurePoints.get(i).getFeaturePointType();
            if (wMap.containsKey(fpt)) {
                featurePointTypes.put(fpt, fpSpheres.getSize(i));
            }
        }
              
        /*
        this.featurePointTypes = visitor.getFeaturePointWeights()
                .get(getSecondaryDrawableFace().getHumanFace()) // Get FP weights for the secondary face
                .entrySet()
                .stream()
                .map(fpWeights -> Map.entry(
                        fpWeights.getKey(),  // For each FP type at the secondary face...
                        fpWeights.getValue() // ... compute average FP weight over all its facets
                                .values()
                                .stream()
                                .mapToDouble(Double::doubleValue)
                                .average()
                                .orElse(Double.NaN)))
                .filter(fpWeight -> !Double.isNaN(fpWeight.getValue())) // Filter out feature points with Double.NaN weight
                .collect(Collectors.toMap(
                        Map.Entry::getKey,
                        fpType -> DrawableFpWeights.FPW_DEFAULT_SIZE));
        */
    }
    
    private void setVisibleSpheres() {
        List<FeaturePoint> featurePoints = getSecondaryFeaturePoints().getFeaturePoints();
        switch (this.weightedFPsShow) {
            case DistancePanel.SEL_SHOW_ALL_SPHERES:
                for (int i = 0; i < featurePoints.size(); i++) {
                    fpSpheres.resetAllRenderModesToDefault();
                }
                fpSpheres.show(true);
                break;
            case DistancePanel.SEL_SHOW_NO_SPHERES:
                fpSpheres.show(false);
                break;
            case DistancePanel.SEL_SHOW_SELECTED_SPHERES:
                for (int i = 0; i < featurePoints.size(); i++) {
                    if (featurePointTypes.containsKey(featurePoints.get(i).getFeaturePointType())) {
                        fpSpheres.setRenderMode(i, DrawableFpWeights.FPW_DEFAULT_RENDERING_MODE);
                    } else {
                        fpSpheres.setRenderMode(i, GL2.GL_NEVER);
                    }
                }
                fpSpheres.show(true);
                break;
            default:
        }
        
    }
}
