package cz.fidentis.analyst.tests;

import cz.fidentis.analyst.face.HumanFace;
import cz.fidentis.analyst.face.HumanFaceUtils;
import cz.fidentis.analyst.distance.HausdorffDistance;
import cz.fidentis.analyst.sampling.NoSampling;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Duration;
import java.util.List;
import java.util.stream.Collectors;

/**
 * A class for testing the efficiency of batch processing algorithms.
 * The goal of this tool is to create a "ground truth" measurements for other optimization techniques
 * <ul>
 * <li>All pairs of faces are taken one by one from the collection.</li>
 * <li>ICP registration is performed for each pair, but only in one direction (the second to the first)</li>
 * <li>Hausdorff distance for each pair is computed in both direction (A-B and B-A). HD uses POINT_TO_POINT strategy and absolute distances.</li>
 * </ul>
 * Stats for 100 faces WITH CROP:
 * <pre>
 * ICP computation time: 02:27:32,170
 * HD computation time: 02:08:34,001
 * Total computation time: 05:29:55,321
 * </pre>
 * Stats for 100 faces WITHOUT CROP:
 * <pre>
 * ICP computation time: 02:23:06,495
 * HD computation time: 01:59:55,520
 * Total computation time: 05:16:46,154
 * </pre>
 * 
 * @author Radek Oslejsek
 */
public class BatchSimilarityGroundTruth {
    
    private static final String DATA_DIR = "../../analyst-data-antropologie/_ECA";
    private static final String OUTPUT_FILE = "../../SIMILARITY_GROUND_TRUTH.csv";
    private static final int MAX_SAMPLES = 100;
    private static final boolean CROP_HD = false;
    
    /**
     * Main method 
     * @param args Input arguments 
     * @throws IOException on IO error
     */
    public static void main(String[] args) throws IOException, ClassNotFoundException, Exception {
        List<Path> faces = Files.list(new File(DATA_DIR).toPath())
                .filter(f -> f.toString().endsWith(".obj"))
                .sorted()
                .limit(MAX_SAMPLES)
                .collect(Collectors.toList());
        
        double[][] distances = new double[faces.size()][faces.size()];
        String[] names = new String[faces.size()];
        
        long totalTime = System.currentTimeMillis();
        long icpComputationTime = 0;
        long hdComputationTime = 0;
        
        int counter = 0;
        for (int i = 0; i < faces.size(); i++) {
            HumanFace priFace = new HumanFace(faces.get(i).toFile());
            names[i] = priFace.getShortName().replaceAll("_01_ECA", "");
            
            for (int j = i; j < faces.size(); j++) { // starts with "i"!
                HumanFace secFace = new HumanFace(faces.get(j).toFile());
                
                System.out.println(++counter + ": " + priFace.getShortName() + " - " + secFace.getShortName());
                
                // transform secondary face
                if (i != j) { // register only different faces
                    long icpTime = System.currentTimeMillis();
                    HumanFaceUtils.alignMeshes(
                            priFace,
                            secFace, // is transformed
                            100,  // max iterations
                            false,// scale
                            0.3,  // error
                            new NoSampling(),  // no undersampling
                            false // drop k-d tree, if exists
                    );
                    icpComputationTime += System.currentTimeMillis() - icpTime;
                }
                
                long hdTime = System.currentTimeMillis();
                // compute HD from secondary to primary:
                HausdorffDistance hd = new HausdorffDistance(
                        priFace.getKdTree(), 
                        HausdorffDistance.Strategy.POINT_TO_POINT,
                        false,  // relative distance
                        true,   // parallel
                        CROP_HD // crop
                );
                secFace.getMeshModel().compute(hd);
                distances[j][i] = hd.getStats().getAverage();

                // compute HD from primary to secondary:
                hd = new HausdorffDistance(
                        secFace.getKdTree(), 
                        HausdorffDistance.Strategy.POINT_TO_POINT,
                        false,  // relative distance
                        true,   // parallel
                        CROP_HD // crop
                );
                priFace.getMeshModel().compute(hd);
                distances[i][j] = hd.getStats().getAverage();

                hdComputationTime += System.currentTimeMillis() - hdTime;
            }
        }
        
        BufferedWriter w = new BufferedWriter(new FileWriter(OUTPUT_FILE));
        w.write("PRI FACE;SEC FACE;AVG HD from PRI to SEC;AVG HD from SEC to PRI;AVG HD lower; AVG HD higher");
        w.newLine();
        for (int i = 0; i < faces.size(); i++) {
            for (int j = i; j < faces.size(); j++) {
                w.write(names[i] + ";");
                w.write(names[j] + ";");
                w.write(String.format("%.8f", distances[i][j]) + ";");
                w.write(String.format("%.8f", distances[j][i]) + ";");
                if (distances[i][j] > distances[j][i]) {
                    w.write(String.format("%.8f", distances[i][j]) + ";");
                    w.write(String.format("%.8f", distances[j][i]) + "");
                } else {
                    w.write(String.format("%.8f", distances[j][i]) + ";");
                    w.write(String.format("%.8f", distances[i][j]) + "");
                }
                w.newLine();
            }
        }
        w.close();
        
        System.out.println();
        Duration duration = Duration.ofMillis(icpComputationTime);
        System.out.println("ICP computation time: " + 
                String.format("%02d:%02d:%02d,%03d", duration.toHoursPart(), duration.toMinutesPart(), duration.toSecondsPart(), duration.toMillisPart()));
        duration = Duration.ofMillis(hdComputationTime);
        System.out.println("HD computation time: " + 
                String.format("%02d:%02d:%02d,%03d", duration.toHoursPart(), duration.toMinutesPart(), duration.toSecondsPart(), duration.toMillisPart()));
        duration = Duration.ofMillis(System.currentTimeMillis() - totalTime);
        System.out.println("Total computation time: " + 
                String.format("%02d:%02d:%02d,%03d", duration.toHoursPart(), duration.toMinutesPart(), duration.toSecondsPart(), duration.toMillisPart()));
        
    }

}
