package cz.fidentis.analyst.featurepoints;

import cz.fidentis.analyst.canvas.Canvas;
import cz.fidentis.analyst.core.ControlPanelAction;
import cz.fidentis.analyst.core.DoubleSpinner;
import cz.fidentis.analyst.core.LoadedActionEvent;
import cz.fidentis.analyst.events.HumanFaceEvent;
import cz.fidentis.analyst.events.HumanFaceListener;
import cz.fidentis.analyst.feature.FeaturePoint;
import cz.fidentis.analyst.feature.FeaturePointType;
import cz.fidentis.analyst.feature.RelationToMesh;
import java.awt.Color;
import java.awt.event.ActionEvent;
import java.util.ArrayList;
import java.util.List;
import javax.swing.JTabbedPane;
import javax.swing.JToggleButton;

/**
 *
 * @author Katerina Zarska
 */
public class FeaturePointsAction extends ControlPanelAction implements HumanFaceListener {
    
    /*
     * GUI elements
     */
    private FeaturePointType hoveredFeaturePoint = null;
    private final FeaturePointsPanel controlPanel;
    private List<FeaturePointType> selectedFeaturePoints = new ArrayList<>();;
    
    private static final Color  FEATURE_POINT_HIGHLIGHT_COLOR = Color.MAGENTA;
    private static final Color  FEATURE_POINT_HOVER_COLOR = Color.CYAN;

    /**
     * Constructor
     * @param canvas            OpenGLCanvas
     * @param topControlPanel   Top component for placing control panels
     */
    public FeaturePointsAction(Canvas canvas, JTabbedPane topControlPanel) {
        super(canvas, topControlPanel);        
        
        controlPanel = new FeaturePointsPanel(this, getPrimaryFeaturePoints());

        // Place control panel to the topControlPanel
        topControlPanel.addTab(controlPanel.getName(), controlPanel.getIcon(), controlPanel);
        topControlPanel.addChangeListener(e -> {
            // If the FeaturePoints panel is focused...
            if (((JTabbedPane) e.getSource()).getSelectedComponent() instanceof FeaturePointsPanel) {
                getCanvas().getScene().setDefaultColors();
            }
        });
        
        // Be informed about changes in faces perfomed by other GUI elements
        getPrimaryDrawableFace().getHumanFace().registerListener(this);
        if (getSecondaryDrawableFace() != null) {
            getSecondaryDrawableFace().getHumanFace().registerListener(this);
        }
    }

    @Override
    public void actionPerformed(ActionEvent ae) {
        String action = ae.getActionCommand();
        
        switch (action) {
            case FeaturePointsPanel.ACTION_COMMAND_ADD_FEATURE_POINT:
                break;
            case FeaturePointsPanel.ACTION_COMMAND_REMOVE_FEATURE_POINT:
                workWithSelectedFp("remove");
                controlPanel.getFeaturePointListPanel().refreshPanel(this, getPrimaryFeaturePoints(), selectedFeaturePoints);
                break;
            case FeaturePointsPanel.ACTION_COMMAND_EDIT_FEATURE_POINT:
                break;
            case FeaturePointsPanel.ACTION_COMMAND_EXPORT_FEATURE_POINTS:
                break;
            case FeaturePointsPanel.ACTION_COMMAND_LOAD_FEATURE_POINTS:
                break;
            case FeaturePointListPanel.ACTION_COMMAND_FEATURE_POINT_SELECT:
                selectFeaturePoint((LoadedActionEvent) ae);
                highlightSelectedFeaturePoints();
                break;
            case FeaturePointListPanel.ACTION_COMMAND_FEATURE_POINT_HOVER_IN:
                hoverFeaturePoint((LoadedActionEvent) ae, true);
                break;
            case FeaturePointListPanel.ACTION_COMMAND_FEATURE_POINT_HOVER_OUT:
                hoverFeaturePoint((LoadedActionEvent) ae, false);
                break;
            case FeaturePointsPanel.ACTION_CHANGE_DISTANCE_THRESHOLD:
                changeDistanceThreshold(ae);
                break; 
            case FeaturePointsPanel.ACTION_PIN_ALL_FEATURE_POINTS:
                for (FeaturePoint fp : getPrimaryFeaturePoints().getFeaturePoints()) {
                    if (fp.getRelationToMesh().getDistance() != 0) {
                        pinFeaturePointToMesh(fp);
                    }
                }
                controlPanel.getFeaturePointListPanel().refreshPanel(this, getPrimaryFeaturePoints(), selectedFeaturePoints);
                break;
            case FeaturePointsPanel.ACTION_PIN_SELECTED_FEATURE_POINTS:
                workWithSelectedFp("pin");
                controlPanel.getFeaturePointListPanel().refreshPanel(this, getPrimaryFeaturePoints(), selectedFeaturePoints);
                break;
            default:
                // do nothing
        }
        renderScene();
    }
    
    /**
     * Changes the color of the face's feature point at the given index
     * and of its weighted representation when the cursor hovers over the feature point's name.
     * The index is received as the data payload of {@code actionEvent}.
     * 
     * @param actionEvent Action event with the index of the feature point as its payload data
     * @param entered {@code true} if the cursor entered the feature point,
     *                {@code false} if the cursor left the feature point
     */
    private void hoverFeaturePoint(LoadedActionEvent actionEvent, boolean entered) {
        final int index = (int) actionEvent.getData();
        
        if (entered) { // entering a feature point
            getPrimaryFeaturePoints().setColor(index, FEATURE_POINT_HOVER_COLOR);
            hoveredFeaturePoint = getTypeOfFeaturePoint(index);
        } else if (selectedFeaturePoints.contains(hoveredFeaturePoint)) { // leaving highlighted FP
            getPrimaryFeaturePoints().setColor(index, FEATURE_POINT_HIGHLIGHT_COLOR);
        } else { // leaving ordinary FP
            getPrimaryFeaturePoints().resetColorToDefault(index);
        }
    }
    
    /**
     * Returns type of the feature point at the given index.
     * 
     * @param index Index of the feature point
     * @return Type of the feature point or {@code null}
     */
    private FeaturePointType getTypeOfFeaturePoint(int index) {
        final List<FeaturePoint> featurePoints = getPrimaryFeaturePoints().getFeaturePoints();
        if (index < 0 || index >= featurePoints.size()) {
            return null;
        }
        
        return featurePoints.get(index).getFeaturePointType();
    }
    
    /**
     * Adds a point to selected feature points
     * 
     * @param actionEvent LoadedActionEvent
     */
    private void selectFeaturePoint(LoadedActionEvent actionEvent) {
        final int index = (int) actionEvent.getData();
        final FeaturePointType fpType = getTypeOfFeaturePoint(index);
        
        if (((JToggleButton) actionEvent.getSource()).isSelected()) {
            selectedFeaturePoints.add(fpType);
        } else {
            selectedFeaturePoints.remove(fpType);
        }
    }
    
    private void highlightSelectedFeaturePoints() {
        if (selectedFeaturePoints.isEmpty()) {
            return;
        }

        for (int i = 0; i < getPrimaryFeaturePoints().getFeaturePoints().size(); i++) {
            if (selectedFeaturePoints.contains(getTypeOfFeaturePoint(i))) {
                getPrimaryFeaturePoints().setColor(i, FEATURE_POINT_HIGHLIGHT_COLOR);
            } else {
                getPrimaryFeaturePoints().resetColorToDefault(i);
            }
        }
    }
    
    private void workWithSelectedFp(String action) {
        if (selectedFeaturePoints.isEmpty()) {
            return;
        }

        for (int i = getPrimaryFeaturePoints().getFeaturePoints().size()-1; i >= 0; i--) {
            if (selectedFeaturePoints.contains(getTypeOfFeaturePoint(i))) {
                switch(action) {
                    case "remove":
                        getPrimaryFeaturePoints().resetColorToDefault(i);
                        getPrimaryFeaturePoints().getFeaturePoints().remove(i);
                        break;
                    case "pin":
                        pinFeaturePointToMesh(getPrimaryFeaturePoints().getFeaturePoints().get(i));
                        break;
                    default:
                        throw new IllegalArgumentException("action " + action);
                }
            }
        }
    }
         
    /**
     * Changes the distance threshold for categorizing feature points into either ON_THE_MESH or CLOSE_TO_MESH
     * 
     * @param ae ActionEvent
     */
    private void changeDistanceThreshold(ActionEvent ae) {
        double sliderVal = (Double) ((DoubleSpinner) ae.getSource()).getValue();
        RelationToMesh.setDistanceThreshold(sliderVal);
        controlPanel.getFeaturePointListPanel().refreshPanel(this, getPrimaryFeaturePoints(), selectedFeaturePoints);
    }
      
    /**
     * Changes the feature point's position to the closest point on mesh
     * 
     * @param fp feature point to be changed
     */
    private void pinFeaturePointToMesh(FeaturePoint fp) {
        RelationToMesh relation = fp.getRelationToMesh();
        if (relation.getPositionType() != RelationToMesh.FpPositionType.OFF_THE_MESH) {
            fp.setPosition(relation.getNearestPoint());
            relation.setDistance(0);
        }
    }  

    @Override
    public void acceptEvent(HumanFaceEvent event) {        
    }
}
