package cz.fidentis.analyst.mesh.core;

import java.util.ArrayList;
import java.util.List;

/**
 * Representation of mesh in memory using corner table
 *
 * @author Matej Lukes
 */
public class CornerTable {

    private List<CornerTableRow> rows = new ArrayList<>();

    /**
     * returns index of face that contains corner.
     * returns -2 if index is less than 0 or more than number of rows in corner table
     *
     * @param index index of corner
     * @return index of corner
     */
    public int getIndexOfFace(int index) {
        if (index < 0 || index > rows.size())
            return -2;
        return index / 3;
    }

    /**
     * returns index of opposite corner.
     * returns -1 if corner does not have opposite
     * returns -2 if index is less than 0 or more than number of rows in corner table
     *
     * @param index index of corner
     * @return index of corner
     */
    public int getIndexOfOppositeCorner(int index) {
        if (index < 0 || index > rows.size())
            return -2;
        return rows.get(index).getOppositeCornerIndex();
    } // opposite

    /**
     * returns index of next corner in Face
     * returns -2 if index is less than 0 or more than number of rows in corner table
     *
     * @param index index of corner
     * @return index of next corner
     */
    public int getIndexOfNextCornerInFace(int index) {
        if (index < 0 || index > rows.size())
            return -2;
        if ((index % 3) == 2)
            return index - 1;
        return index + 1;
    }

    /**
     * returns index of previous corner in Face
     * returns -2 if index is less than 0 or more than number of rows in corner table
     *
     * @param index index of corner
     * @return index of previous corner
     */
    public int getIndexOfPreviousCornerInFace(int index) {
        if (index < 0 || index > rows.size())
            return -2;
        return getIndexOfNextCornerInFace(getIndexOfNextCornerInFace(index));
    }

    /**
     * returns opposite corner of edge on the left side of corner
     * returns -2 if index is less than 0 or more than number of rows in corner table
     *
     * @param index index of corner
     * @return index of corner
     */
    public int getIndexOfTipCornerOnLeft(int index) {
        if (index < 0 || index > rows.size())
            return -2;
        return getIndexOfOppositeCorner(getIndexOfPreviousCornerInFace(index));
    }

    /**
     * returns opposite corner of edge on the right side of corner
     * returns -2 if index is less than 0 ord more than number of rows in corner table
     *
     * @param index index of corner
     * @return index of corner
     */
    public int getIndexOfTipCornerOnRight(int index) {
        if (index < 0 || index > rows.size())
            return -2;
        return getIndexOfOppositeCorner(getIndexOfNextCornerInFace(index));
    }

    /**
     * returns corner next to the corner opposite the edge on the left side
     * returns -2 if index is less than 0 ord more than number of rows in corner table
     *
     * @param index index of corner
     * @return index of corner
     */
    public int getNextAroundCorner(int index) {
        if (index < 0 || index > rows.size())
            return -2;
        return getIndexOfNextCornerInFace(getIndexOfTipCornerOnLeft(index));
    }

    /**
     * adds row at the end of corner table
     *
     * @param row that is going to be added to corner table
     */
    public void addRow(CornerTableRow row) {
        rows.add(row);
    }

    /**
     * replaces row of corner table at specified index
     *
     * @param index index of replaced row
     * @param row new row
     */
    public void replaceRow(int index, CornerTableRow row) {
        rows.set(index, row);
    }
}
