package cz.fidentis.analyst.mesh.io;

import com.mokiat.data.front.parser.IOBJParser;
import com.mokiat.data.front.parser.OBJDataReference;
import com.mokiat.data.front.parser.OBJFace;
import com.mokiat.data.front.parser.OBJMesh;
import com.mokiat.data.front.parser.OBJModel;
import com.mokiat.data.front.parser.OBJNormal;
import com.mokiat.data.front.parser.OBJObject;
import com.mokiat.data.front.parser.OBJParser;
import com.mokiat.data.front.parser.OBJTexCoord;
import com.mokiat.data.front.parser.OBJVertex;
//import cz.fidentis.analyst.mesh.core.MeshEdge;
//import cz.fidentis.analyst.mesh.core.MeshFace;
import cz.fidentis.analyst.mesh.core.MeshFacet;
import cz.fidentis.analyst.mesh.core.MeshModel;
import cz.fidentis.analyst.mesh.core.MeshPoint;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import javax.vecmath.Vector3d;


/**
 * Utility class for loading human face from OBJ data format.
 * Expects only one human face in one file stored as one object.
 * @author Marek Bařinka
 */
public class MeshObjLoader {
    
    /**
     * Loads data into MeshModel
     * @param is Input data stream with model
     * @return Complete MeshModel
     * @throws IOException Data are corrupted
     */
    public static MeshModel read(InputStream is) throws IOException {
        final IOBJParser parser = new OBJParser();
        final OBJModel model = parser.parse(is);

        if (model.getObjects().isEmpty()) {
            throw new IOException("File doesn't contain any model");
        }
        OBJObject object = model.getObjects().get(0); 
        return parseObjectToModel(model, object);
    }
    
    /**
     * Opens file and loads data into MeshModel
     * @param file File containing face to load into a MeshModel
     * @return Complete MeshModel
     * @throws FileNotFoundException Requested file was not found
     * @throws IOException There was problem with reading the file
     */
    public static MeshModel read(File file) throws FileNotFoundException, IOException {
        try (InputStream is = new FileInputStream(file)) {
            return read(is);
        }
    }    
    
    /**
     * Parse OBJObject into MeshModel
     * @param model Model is needed in future. It's holding data pools
     * @param object Object to parse. It corespond to our MeshModel
     * @return Returns complete model
     * @throws IOException Data are corrupted
     */
    private static MeshModel parseObjectToModel(OBJModel model, OBJObject object) throws IOException {
        MeshModel meshModel = new MeshModel();
        // Our facet = loader mesh, create and fill all facets
        for (OBJMesh mesh : object.getMeshes()) {
            MeshFacet meshFacet = parseMeshToFacet(model, mesh);
//            meshModel.addChild(meshFacet);
        }
        return meshModel;
    }
    
    /**
     * Parse OBJMesh into MeshFacet
     * @param model Model is needed in future. It's holding data pools
     * @param mesh Mesh to parse. It corespond to our MeshMacet
     * @return Returns complete facet
     * @throws IOException Data are corrupted
     */
    private static MeshFacet parseMeshToFacet(OBJModel model, OBJMesh mesh) throws IOException {
        MeshFacet meshFacet = new MeshFacet();

        for (OBJFace face : mesh.getFaces()) {
//            MeshFace meshFace = parseFaceToFace(model, face, meshFacet);
//            meshFacet.addChild(meshFace);
        }
        return meshFacet;
    }
    
    /**
     * Parse OBJFace into MeshFace
     * @param model Model is needed in future. It's holding data pools
     * @param face Face to parse. It corespond to our MeshFace
     * @param meshFacet Facet is holding our data pools
     * @return Returns complete face
     * @throws IOException Data are corrupted
     */
    /*
    private static MeshFace parseFaceToFace(OBJModel model, OBJFace face, MeshFacet meshFacet) throws IOException {
        List<MeshPoint> facePoints = getPointsFromFace(model, face, meshFacet);
        MeshFace meshFace = new MeshFace();

        if (facePoints.size() != 3) {
            throw new IOException("Data contains non-triangular face");
        }
        // Create edges
        for (int i = 1; i <= facePoints.size(); i++) {
            MeshEdge meshEdge = new MeshEdge(facePoints.get((i-1)%facePoints.size()), facePoints.get((i)%facePoints.size()));
            meshEdge = meshFacet.getEdge(meshEdge);
            meshFace.addChild(meshEdge);
        }
        return meshFace;
    }
    */
    
    /**
     * Creates list of points from one face
     * @param model Model holds points pool
     * @param face Face for which points are loaded
     * @param meshFacet Facet holds points pool
     * @return List of all points from one face
     */
    private static List<MeshPoint> getPointsFromFace(OBJModel model, OBJFace face, MeshFacet meshFacet) {
        List<OBJDataReference> references = face.getReferences();
        List<MeshPoint> points = new ArrayList();
        
        for (OBJDataReference reference : references) {
            final OBJVertex vertex = model.getVertex(reference);
            Vector3d coords = new Vector3d(vertex.x, vertex.y, vertex.z);
            Vector3d norm = null;
            Vector3d texCoords = null;
            if (reference.hasNormalIndex()) {
                final OBJNormal normal = model.getNormal(reference);
                norm = new Vector3d(normal.x, normal.y, normal.z);
            }
            if (reference.hasTexCoordIndex()) {
                final OBJTexCoord texCoord = model.getTexCoord(reference);
                texCoords = new Vector3d(texCoord.u, texCoord.v, texCoord.w);
            }
//            MeshPoint meshPoint = meshFacet.getPoint(new MeshPoint(coords, norm, texCoords));
//            points.add(meshPoint);                            
        }
        return points;
    }
}