package cz.fidentis.analyst.distance;

import com.jogamp.opengl.GL2;
import cz.fidentis.analyst.canvas.Canvas;
import cz.fidentis.analyst.core.LoadedActionEvent;
import cz.fidentis.analyst.core.ControlPanelAction;
import cz.fidentis.analyst.feature.FeaturePoint;
import cz.fidentis.analyst.feature.FeaturePointType;
import cz.fidentis.analyst.mesh.core.MeshFacet;
import cz.fidentis.analyst.scene.DrawableFeaturePoints;
import cz.fidentis.analyst.visitors.face.HausdorffDistancePrioritized;
import cz.fidentis.analyst.visitors.mesh.HausdorffDistance.Strategy;
import java.awt.Color;
import java.awt.event.ActionEvent;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import javax.swing.JComboBox;
import javax.swing.JTabbedPane;
import javax.swing.JTextField;
import javax.swing.JToggleButton;
import javax.vecmath.Point3d;

/**
 * Action listener for computation of the Hausdorff distance.
 * 
 * @author Radek Oslejsek
 * @author Daniel Schramm
 */
public class DistanceAction extends ControlPanelAction {
    
    /*
     * Attributes handling the state
     */
    private HausdorffDistancePrioritized visitor = null;
    private final Map<FeaturePointType, Double> featurePointTypes;
    private String strategy = DistancePanel.STRATEGY_POINT_TO_POINT;
    private boolean relativeDist = false;
    private boolean weightedDist = false;
    private boolean heatmapRender = false;
    private boolean weightedFPsShow = true;
    
    private Map<MeshFacet, List<Double>> weightedHausdorffDistance = null;
    
    private final DistancePanel controlPanel;
    private final DrawableFeaturePoints weightedFeaturePoints;
    
    private static final Color FEATURE_POINT_HIGHLIGHT_COLOR = Color.MAGENTA;
    
    /**
     * Constructor.
     * 
     * @param canvas OpenGL canvas
     * @param topControlPanel Top component for placing control panels
     */
    public DistanceAction(Canvas canvas, JTabbedPane topControlPanel) {
        super(canvas, topControlPanel);
        
        // Calculate weighted Hausdorff distance for all feature points of the secondary face
        calculateHausdorffDistance(
                getSecondaryFeaturePoints().getFeaturePoints()
                        .stream()
                        .collect(Collectors.toMap(
                                FeaturePoint::getFeaturePointType,
                                featurePoint -> DrawableFeaturePoints.DEFAULT_SIZE))
        );
        
        this.featurePointTypes = visitor.getFeaturePointWeights()
                .get(getSecondaryDrawableFace().getHumanFace()) // Get FP weights for the secondary face
                .entrySet()
                .stream()
                .map(fpWeights -> Map.entry(
                        fpWeights.getKey(),  // For each FP type at the secondary face...
                        fpWeights.getValue() // ... compute average FP weight over all its facets
                                .values()
                                .stream()
                                .mapToDouble(weight -> weight)
                                .average()
                                .orElse(Double.NaN)))
                .filter(fpWeight -> !Double.isNaN(fpWeight.getValue())) // Filter out feature points with Double.NaN weight
                .collect(Collectors.toMap(
                        Map.Entry::getKey,
                        fpType -> DrawableFeaturePoints.DEFAULT_SIZE));
        this.controlPanel = new DistancePanel(this, getSecondaryFeaturePoints().getFeaturePoints(), featurePointTypes.keySet());
        this.visitor = null;
        
        // Add weighted feature points to the scene
        this.weightedFeaturePoints = new DrawableFeaturePoints(getSecondaryFeaturePoints().getFeaturePoints());
        weightedFeaturePoints.setRenderMode(GL2.GL_LINE);
        weightedFeaturePoints.setColor(Color.WHITE);
        getCanvas().getScene().addOtherDrawable(weightedFeaturePoints);
        
        // Place control panel to the topControlPanel
        topControlPanel.addTab(controlPanel.getName(), controlPanel.getIcon(), controlPanel);
        topControlPanel.addChangeListener(e -> {
            // If the distance panel is focused...
            if (((JTabbedPane) e.getSource()).getSelectedComponent() instanceof DistancePanel) {
                // ... display heatmap and feature points relevant to the Hausdorff distance
                getCanvas().getScene().setDefaultColors();
                weightedFeaturePoints.resetAllColorsToDefault();
                if (weightedFPsShow) {
                    weightedFeaturePoints.show();
                }
                
                boolean secondaryFaceMoved = false;
                final List<FeaturePoint> secondaryFPs = getSecondaryFeaturePoints().getFeaturePoints();
                final List<FeaturePoint> weightedFPs = weightedFeaturePoints.getFeaturePoints();
                for (int i = 0; i < secondaryFPs.size(); i++) {
                    final FeaturePoint faceFP = secondaryFPs.get(i);
                    final FeaturePoint weightedFP = weightedFPs.get(i);
                    
                    // Highlight feature point if selected
                    if (featurePointTypes.containsKey(faceFP.getFeaturePointType())) {
                        getSecondaryFeaturePoints().setColor(i, FEATURE_POINT_HIGHLIGHT_COLOR);
                        weightedFeaturePoints.setColor(i, FEATURE_POINT_HIGHLIGHT_COLOR);
                    }
                    if (!faceFP.getPosition().equals(weightedFP.getPosition())) {
                        secondaryFaceMoved = true;
                    }
                }
                
                // If the positin of secondary face has been changed
                if (secondaryFaceMoved) {
                    visitor = null; // recompute Hausdorff distance
                    
                    // Relocate weighted feature points
                    for (int i = 0; i < secondaryFPs.size(); i++) {
                        final Point3d faceFpPos = secondaryFPs.get(i).getPosition();
                        final Point3d weightedFpPos = weightedFPs.get(i).getPosition();
                        
                        weightedFpPos.x = faceFpPos.x;
                        weightedFpPos.y = faceFpPos.y;
                        weightedFpPos.z = faceFpPos.z;
                    }
                }
                
                updateHausdorffDistanceInformation();
                getSecondaryDrawableFace().setRenderHeatmap(heatmapRender);
            } else {
                weightedFeaturePoints.hide();
            }
        });
        topControlPanel.setSelectedComponent(controlPanel); // Focus Hausdorff distance panel
    }

    @Override
    public void actionPerformed(ActionEvent ae) {
        final String action = ae.getActionCommand();
        
        switch (action) {
            case DistancePanel.ACTION_COMMAND_SHOW_HIDE_HEATMAP:
                heatmapRender = ((JToggleButton) ae.getSource()).isSelected();
                if (heatmapRender) {
                    updateHausdorffDistanceInformation();
                }
                getSecondaryDrawableFace().setRenderHeatmap(heatmapRender);
                break;
            case DistancePanel.ACTION_COMMAND_SHOW_HIDE_WEIGHTED_FPOINTS:
                weightedFPsShow = ((JToggleButton) ae.getSource()).isSelected();
                if (weightedFPsShow) {
                    weightedFeaturePoints.show();
                } else {
                    weightedFeaturePoints.hide();
                }
                break;
            case DistancePanel.ACTION_COMMAND_SET_DISTANCE_STRATEGY:
                strategy = (String) ((JComboBox) ae.getSource()).getSelectedItem();
                this.visitor = null; // recompute
                updateHausdorffDistanceInformation();
                break;
            case DistancePanel.ACTION_COMMAND_RELATIVE_ABSOLUTE_DIST:
                this.relativeDist = ((JToggleButton) ae.getSource()).isSelected();
                this.visitor = null; // recompute
                updateHausdorffDistanceInformation();
                break;
            case DistancePanel.ACTION_COMMAND_FEATURE_POINT_HIGHLIGHT:
                highlightFeaturePoint((LoadedActionEvent) ae);
                break;
            case DistancePanel.ACTION_COMMAND_FEATURE_POINT_RESIZE:
                resizeFeaturePoint((LoadedActionEvent) ae);
                break;
            case DistancePanel.ACTION_COMMAND_WEIGHTED_DISTANCE:
                this.weightedDist = ((JToggleButton) ae.getSource()).isSelected();
                updateHausdorffDistanceInformation();
                break;
            case DistancePanel.ACTION_COMMAND_DISTANCE_RECOMPUTE:
                this.visitor = null; // recompute
                updateHausdorffDistanceInformation();
                break;
            default:
                // to nothing
        }
        renderScene();
    }
    
    /**
     * (Re)calculates the Hausdorff distance and updates the heat map of the secondary face
     * as well as values of all appropriate GUI elements of {@link DistancePanel}.
     */
    private void updateHausdorffDistanceInformation() {
        if (visitor == null) {
            calculateHausdorffDistance(featurePointTypes);
            
            weightedHausdorffDistance = getWeightedDistance();
            
            // Update GUI elements that display the calculated Hausdorff distance metrics
            setFeaturePointWeigths();
            setHausdorffDistanceStatistics();
        }
        
        getSecondaryDrawableFace().setHeatMap(weightedDist ? weightedHausdorffDistance : visitor.getDistances());
    }
    
    /**
     * (Re)calculates the Hausdorff distance.
     * 
     * @param featurePoints Types of feature points according to which the Hausdorff
     *                      distances will be prioritized together with
     *                      the radii of priority spheres around the feature
     *                      points of the corresponding type.
     *                      Must not be {@code null}.
     */
    private void calculateHausdorffDistance(Map<FeaturePointType, Double> featurePoints) {
        final Strategy useStrategy;
        switch (strategy) {
            case DistancePanel.STRATEGY_POINT_TO_POINT:
                useStrategy = Strategy.POINT_TO_POINT;
                break;
            case DistancePanel.STRATEGY_POINT_TO_TRIANGLE:
                useStrategy = Strategy.POINT_TO_TRIANGLE_APPROXIMATE;
                break;
            default:
                throw new UnsupportedOperationException(strategy);
        }
        
        this.visitor = new HausdorffDistancePrioritized(getPrimaryDrawableFace().getModel(),
                featurePoints,
                useStrategy,
                relativeDist,
                true);
        getSecondaryDrawableFace().getHumanFace().accept(visitor);
    }
    
    /**
     * Calculates weighted Hausdorff distance of the face.
     * 
     * @return weighted Hausdorff distance
     */
    private Map<MeshFacet, List<Double>> getWeightedDistance() {
        final Map<MeshFacet, List<Double>> weightedDistances = new HashMap<>(visitor.getDistances());
        final Map<MeshFacet, List<Double>> mergedPriorities = visitor.getMergedPriorities()
                .get(getSecondaryDrawableFace().getHumanFace());

        // Merge the map of distances with the map of priorities
        for (final Map.Entry<MeshFacet, List<Double>> facetPriorities: mergedPriorities.entrySet()) {
            weightedDistances.merge(
                    facetPriorities.getKey(),
                    facetPriorities.getValue(),
                    (distancesList, prioritiesList) ->
                            IntStream.range(0, distancesList.size())
                                    .mapToDouble(i -> distancesList.get(i) * prioritiesList.get(i))
                                    .boxed()
                                    .collect(Collectors.toList()));
        }

        return weightedDistances;
    }
    
    /**
     * Updates the GUI elements of {@link DistancePanel} that display
     * the weights of feature points used to calculate the weighted Hausdorff distance.
     */
    private void setFeaturePointWeigths() {
        controlPanel.updateFeaturePointWeights(
                visitor.getFeaturePointWeights()
                        .get(getSecondaryDrawableFace().getHumanFace()) // Get FP weights for the secondary face
                        .entrySet()
                        .stream()
                        .collect(
                                Collectors.toMap(
                                        Map.Entry::getKey,            // For each FP type at the secondary face...
                                        weights -> weights.getValue() // ... compute average FP weight over all its facets
                                                .values()
                                                .stream()
                                                .mapToDouble(weight -> weight)
                                                .average()
                                                .orElse(Double.NaN))));
    }

    /**
     * Updates the GUI elements of {@link DistancePanel} elements that display
     * statistical data about the calculated Hausdorff distance.
     */
    private void setHausdorffDistanceStatistics() {
        controlPanel.updateHausdorffDistanceStats(
                visitor.getDistances()
                        .values()
                        .stream()
                        .flatMap(List::stream)
                        .mapToDouble(distance -> distance)
                        .summaryStatistics(),
                weightedHausdorffDistance
                        .values()
                        .stream()
                        .flatMap(List::stream)
                        .mapToDouble(distance -> distance)
                        .summaryStatistics()
        );
    }
    
    /**
     * Changes the colour of the secondary face's feature point at the given index.
     * The index is received as the data payload of {@code actionEvent}.
     * 
     * @param actionEvent Action event with the index of the feature point as its payload data
     */
    private void highlightFeaturePoint(LoadedActionEvent actionEvent) {
        final int index = (int) actionEvent.getData();
        final FeaturePointType fpType = getTypeOfFeaturePoint(index);
        final DrawableFeaturePoints secondaryFeaturePoints = getSecondaryFeaturePoints();
        
        if (((JToggleButton) actionEvent.getSource()).isSelected()) {
            secondaryFeaturePoints.setColor(index, FEATURE_POINT_HIGHLIGHT_COLOR);
            weightedFeaturePoints.setColor(index, FEATURE_POINT_HIGHLIGHT_COLOR);
            featurePointTypes.put(fpType, weightedFeaturePoints.getSize(index));
        } else {
            secondaryFeaturePoints.resetColorToDefault(index);
            weightedFeaturePoints.resetColorToDefault(index);
            featurePointTypes.remove(fpType);
        }
    }

    /**
     * Changes the size of the secondary face's feature point at the given index.
     * The index is received as the data payload of {@code actionEvent}.
     * 
     * @param actionEvent Action event with the index of the feature point as its payload data
     */
    private void resizeFeaturePoint(LoadedActionEvent actionEvent) {
        final int index = (int) actionEvent.getData();
        final double size = Double.parseDouble(((JTextField) actionEvent.getSource()).getText());
        
        weightedFeaturePoints.setSize(index, size);
        featurePointTypes.replace(getTypeOfFeaturePoint(index), size);
    }
    
    /**
     * Returns type of the feature point at the given index in the secondary face.
     * 
     * @param index Index of the feature point
     * @return Type of the feature point or {@code null}
     */
    private FeaturePointType getTypeOfFeaturePoint(int index) {
        final List<FeaturePoint> featurePoints = getSecondaryFeaturePoints().getFeaturePoints();
        if (index < 0 || index >= featurePoints.size()) {
            return null;
        }
        
        return featurePoints.get(index)
                .getFeaturePointType();
    }
}
