package cz.fidentis.analyst.feature.utils;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.util.List;

/**
 * Class working with files
 * 
 * @author Jakub Kolman
 */
public class FileResourcesUtils {

    /**
     * main method
     * 
     * @param args
     * @throws IOException
     * @throws URISyntaxException 
     */
//    public static void main(String[] args) throws IOException, URISyntaxException {
//
//        FileResourcesUtils app = new FileResourcesUtils();
//
//        //String fileName = "database.properties";
//        String fileName = "json/file1.json";
//
//        System.out.println("getResourceAsStream : " + fileName);
//        InputStream is = app.getFileFromResourceAsStream(fileName);
//        printInputStream(is);
//
//        System.out.println("\ngetResource : " + fileName);
//        File file = app.getFileFromResource(fileName);
//        printFile(file);
//
//    }

    /**
     * get a file from the resources folder
     * works everywhere, IDEA, unit test and JAR file.
     * 
     * @param fileName
     * @return 
     */
    public InputStream getFileFromResourceAsStream(String fileName) {

        // The class loader that loaded the class
        ClassLoader classLoader = getClass().getClassLoader();
        InputStream inputStream = classLoader.getResourceAsStream(fileName);

        // the stream holding the file content
        if (inputStream == null) {
            throw new IllegalArgumentException("file not found! " + fileName);
        } else {
            return inputStream;
        }

    }

    /**
     * The resource URL is not working in the JAR If we try to access a file
     * that is inside a JAR, It throws NoSuchFileException (linux),
     * InvalidPathException (Windows)
     *
     * Resource URL Sample: file:java-io.jar!/json/file1.json
     *
     * @param fileName
     */
    public File getFileFromResource(String fileName) throws URISyntaxException {

        ClassLoader classLoader = getClass().getClassLoader();
        URL resource = classLoader.getResource(fileName);
        if (resource == null) {
            throw new IllegalArgumentException("file not found! " + fileName);
        } else {

            // failed if files have whitespaces or special characters
            //return new File(resource.getFile());
            return new File(resource.toURI());
        }

    }

    /**
     * get a file from the resources folder works everywhere, IDEA, unit test
     * and JAR file.
     *
     * @param path
     * @param fileName
     * @return
     * @throws FileNotFoundException
     */
    public InputStream getFileAsStream(String path, String fileName) throws FileNotFoundException {
        return new FileInputStream(getFile(path, fileName));
    }

    /**
     * Creates input stream from file
     *
     * @param file
     * @return
     * @throws FileNotFoundException
     */
    public InputStream getFileAsStream(File file) throws FileNotFoundException {
        return new FileInputStream(file);
    }

    /**
     * get File from path, file name
     *
     * @param path
     * @param fileName
     * @return
     */
    public File getFile(String path, String fileName) {
        return new File(path, fileName);
    }

    /**
     * print input stream
     *
     * @param is
     */
    private static void printInputStream(InputStream is) {
        try (InputStreamReader streamReader
                = new InputStreamReader(is, StandardCharsets.UTF_8);
                BufferedReader reader = new BufferedReader(streamReader)) {
            String line;
            while ((line = reader.readLine()) != null) {
                System.out.println(line);
            }
        } catch (IOException e) {
            e.printStackTrace();
        }

    }

    // print a file
    private static void printFile(File file) {
        List<String> lines;
        try {
            lines = Files.readAllLines(file.toPath(), StandardCharsets.UTF_8);
            lines.forEach(System.out::println);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

}
