package cz.fidentis.analyst.feature.services;

import cz.fidentis.analyst.feature.FeaturePoint;
import cz.fidentis.analyst.feature.provider.FeaturePointTypeProvider;
import cz.fidentis.analyst.feature.utils.FileResourcesUtils;
import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Stream;

/**
 * Class used to import feature points from file of csv format
 * 
 * @author Jakub Kolman
 */
public class FeaturePointCsvLoader {
    
    private static final String COLUMN_DELIMETER = ",";
    private static final String CODE_PREFIX_DELIMETER = " ";

    /**
     * Loads feature points from file of csv format
     * @param path
     * @param fileName
     * @return
     * @throws FileNotFoundException
     * @throws IOException 
     */
    public static List<FeaturePoint> loadFeaturePoints(String path, String fileName) throws FileNotFoundException, IOException {
        FileResourcesUtils app = new FileResourcesUtils();
        try (InputStreamReader streamReader
                = new InputStreamReader(app.getFileAsStream(path, fileName), StandardCharsets.UTF_8);
                BufferedReader reader = new BufferedReader(streamReader)) {

            Stream<String> lines = reader.lines();
            List<List<String>> linesList = new ArrayList<>();

            lines
                    .forEach(line -> {
                        linesList.add(Arrays.asList(line.split(COLUMN_DELIMETER)));
                    });

            if (linesList.size() != 2
                    || linesList.get(0).size() != linesList.get(1).size()) {
                throw new IOException(String.format("Feature point import file '%s' has wrong format", fileName));
            }

            List<FeaturePoint> points = new ArrayList<>();
            for (int i = 1; i < linesList.get(0).size(); i += 3) {
                FeaturePoint point = new FeaturePoint(
                        Double.parseDouble(linesList.get(1).get(i)),
                        Double.parseDouble(linesList.get(1).get(i + 1)),
                        Double.parseDouble(linesList.get(1).get(i + 2)),
                        FeaturePointTypeProvider.getInstance().getFeaturePointTypeByCode(
                                getCode(linesList.get(0).get(i)))
                );
                points.add(point);
            }
            return points;

        } catch (NumberFormatException e1) {
            throw new IOException(e1);
        }
    }

    private static String getCode(String str) {
        return str.substring(0, str.indexOf(CODE_PREFIX_DELIMETER));
    }
}
