/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package cz.fidentis.analyst.procrustes.utils;

import cz.fidentis.analyst.feature.FeaturePoint;
import cz.fidentis.analyst.feature.provider.FeaturePointTypeProvider;
import cz.fidentis.analyst.mesh.core.MeshPoint;
import cz.fidentis.analyst.procrustes.ProcrustesAnalysisFaceModel;
import org.ejml.simple.SimpleMatrix;

import javax.vecmath.Point3d;
import javax.vecmath.Vector3f;
import java.util.*;

/**
 * @author Jakub Kolman
 */
public class ProcrustesAnalysisUtils {

    /**
     * Sorts List by featurePoint type property
     *
     * @param featurePointList
     * @return ordered list of feature points by type
     */
    public static List<FeaturePoint> sortListByFeaturePointType(List<FeaturePoint> featurePointList) {
        Collections.sort(
                featurePointList, Comparator.comparingInt(fp -> fp.getFeaturePointType().getType())
        );
        return featurePointList;
    }

    /**
     * Creates hash map with a key feature point type and value feature point
     * @param featurePointList
     * @return
     */
    public static HashMap<Integer, FeaturePoint> generateFeaturePointHashMap(List<FeaturePoint> featurePointList) {
        HashMap<Integer, FeaturePoint> map = new HashMap<>();
        for (FeaturePoint fp : featurePointList) {
            if (fp.getFeaturePointType().getType() > 0) {
                map.put(fp.getFeaturePointType().getType(), fp);
            }
        }
        return map;
    }

    /**
     * Checks if two feature point lists have the same types of feature points.
     * <p>
     * To use this method you need to supply ordered lists by Feature Point type
     * as parameters. Otherwise even if two lists contain the same featurepoints
     * it will return false.
     * <p>
     * Use sort function {@link #sortListByFeaturePointType} on feature point lists
     * two get correct results.
     *
     * @param featurePointList1
     * @param featurePointList2
     * @return true if two sorted lists by the feature point type contain the
     * same feature point types
     */
    public static boolean checkFeaturePointsType(List<FeaturePoint> featurePointList1, List<FeaturePoint> featurePointList2) {
        for (int i = 0; i < featurePointList1.size(); i++) {
            if (featurePointList1.get(i).getFeaturePointType().getType() != featurePointList2.get(i).getFeaturePointType().getType()) {
                System.out.print(featurePointList1.get(i).getFeaturePointType().getType());
                System.out.print(featurePointList2.get(i).getFeaturePointType().getType());
                return false;
            }
        }
        return true;
    }

    /**
     * Creates array of distance values from feature point to origin (0,0,0).
     *
     * @param featurePointList
     * @return array of distances
     */
    public static double[] calculateMeanDistancesFromOrigin(List<FeaturePoint> featurePointList) {
        double[] distances = new double[featurePointList.size()];
        for (int i = 0; i < featurePointList.size(); i++) {
            distances[i] = calculatePointDistanceFromOrigin(featurePointList.get(i));
        }
        return distances;
    }

    /**
     * Finds centrioid from given feature point List
     *
     * @param featurePointList
     * @return centroid of feature points (Vector3F)
     */
    public static Point3d findCentroidOfFeaturePoints(List<FeaturePoint> featurePointList) {
        float x = 0;
        float y = 0;
        float z = 0;
        for (FeaturePoint fp : featurePointList) {
            x += fp.getX();
            y += fp.getY();
            z += fp.getZ();
        }
        return new Point3d(x / featurePointList.size(), y / featurePointList.size(), z / featurePointList.size());
    }

    /**
     * Calculates distance of one feature point from another
     *
     * @param fp1
     * @param fp2
     * @return distance
     */
    public static double calculateDistanceOfTwoPoints(FeaturePoint fp1, FeaturePoint fp2) {
        return Math.sqrt(
                (Math.pow(fp1.getX(), 2) - Math.pow(fp2.getX(), 2))
                        + (Math.pow(fp1.getY(), 2) - Math.pow(fp2.getY(), 2))
                        + (Math.pow(fp1.getZ(), 2) - Math.pow(fp2.getZ(), 2)));
    }

    /**
     * Calculates distance of single feature point from origin (0,0,0)
     *
     * @param fp
     * @return distance
     */
    public static double calculatePointDistanceFromOrigin(FeaturePoint fp) {
        return Math.sqrt(
                Math.pow(fp.getX(), 2)
                        + Math.pow(fp.getY(), 2)
                        + Math.pow(fp.getZ(), 2));
    }

    /**
     * Creates matrix from given feature point list
     *
     * @param featurePointList
     * @return matrix
     */
    public static SimpleMatrix createFeaturePointMatrix(List<FeaturePoint> featurePointList) {
        SimpleMatrix matrix = new SimpleMatrix(featurePointList.size(), 3);

        for (int i = 0; i < featurePointList.size(); i++) {
            matrix.set(i, 0, featurePointList.get(i).getX());
            matrix.set(i, 1, featurePointList.get(i).getY());
            matrix.set(i, 2, featurePointList.get(i).getZ());
        }
        return matrix;
    }

    /**
     * Creates feature point map HashMap with key FeaturePoint.type and value FeaturePoint back from matrix.
     *
     * @param matrix
     * @param model
     * @return
     */
    public static HashMap<Integer, FeaturePoint> createFeaturePointMapFromMatrix(SimpleMatrix matrix, ProcrustesAnalysisFaceModel model) {
        HashMap<Integer, FeaturePoint> map = new HashMap<>();
        for (int i = 0; i < matrix.numRows(); i++) {
            FeaturePoint featurePoint = new FeaturePoint(
                    matrix.get(i, 0),
                    matrix.get(i, 1),
                    matrix.get(i, 2),
                    FeaturePointTypeProvider.getInstance().getFeaturePointTypeById(
                            model.getFeaturePointTypeCorrespondence().get(i)));
            map.put(model.getFeaturePointTypeCorrespondence().get(i), featurePoint);
        }
        return map;
    }

    /**
     * Rotates vertex by using matrix multiplication of rotation matrix and vertex position.
     *
     * @param v
     * @param matrix
     * @return rotetedVertex coordinates [x, y, z]
     */
    public static Point3d rotateVertex(MeshPoint v, SimpleMatrix matrix) {
        float x = (float) ((v.getPosition().x * matrix.getIndex(0, 0))
                        + (v.getPosition().y * matrix.getIndex(1, 0))
                        + (v.getPosition().z * matrix.getIndex(2, 0)));
        float y = (float) ((v.getPosition().x * matrix.getIndex(0, 1))
                        + (v.getPosition().y * matrix.getIndex(1, 1))
                        + (v.getPosition().z * matrix.getIndex(2, 1)));
        float z = (float) ((v.getPosition().x * matrix.getIndex(0, 2))
                        + (v.getPosition().y * matrix.getIndex(1, 2))
                        + (v.getPosition().z * matrix.getIndex(2, 2)));
        Point3d rotatedVertex = new Point3d(x, y, z);
        return rotatedVertex;
    }
}
